#ifndef LOCALMAP_SEEN
#define LOCALMAP_SEEN

#include "pixelmap.h"
#include "sphericalmap.h"
#include "ndatablock.h"

#include "anydataobj.h"
#include "ppersist.h"

// A local map of a region of the sky, in cartesian coordinates.
// It has an origin in (theta0, phi0), mapped to pixel(x0, y0)
//   (x0, y0 might be outside of this local map)
// default value of (x0, y0) is middle of the map, center of pixel(nx/2, ny/2)
//
//    la carte est consideree comme un tableau a deux indices i et j, i etant 
//    indice de colonne et j indice de ligne. La carte est supposee resider 
//    dans un plan tangent, dont le point de tangence est repere (x0,y0) dans 
//    la carte et (theta0, phi0) sur la sphere celeste. L extension de la 
//    carte est definie par les valeurs de deux angles couverts respectivement
//    par la totalite des pixels en x de la carte et la totalite des pixels 
//    en y. (SetSize()).
//    On considere un "plan de reference" : plan tangent a la sphere celeste 
//    aux angles theta=Pi/2 et phi=0. Dans ce plan L origine des coordonnees 
//    est le point de tangence. L axe Ox est la tangente parallele a 
//    lequateur, dirige vers les phi croissants, l axe Oy est parallele 
//    au meridien, dirige vers le pole nord.
//    De maniere interne a la classe une carte est definie dans ce plan de 
//    reference et transportee  jusqu au point (theta0, phi0) de sorte que les //    axes restent paralleles aux meridiens et paralleles. L utilisateur peut 
//    definir sa carte selon un repere en rotation par rapport au repere de 
//    reference (par l angle entre le parallele et l axe Ox souhaite -- 
//    methode SetOrigin(...))


// ***************** Class LocalMap *****************************

template<class T>
class LocalMap : public PixelMap<T>, public AnyDataObj 
{

public:

LocalMap();
LocalMap(int_4 nx, int_4 ny);
LocalMap(const LocalMap<T>& lm);
virtual ~LocalMap();

// ---------- Overloading of () to access pixel number k ----

inline T& operator()(int_4 k) {return(PixVal(k));}
inline T const& operator()(int_4 k) const {return(PixVal(k));}
inline T& operator()(int ix, int iy) {return PixVal(iy*nSzX_+ix);};
inline T const& operator()(int ix, int iy) const {return PixVal(iy*nSzX_+ix);};
   
// ---------- Definition of PixelMap abstract methods -------

/* return/set the number of pixels */
virtual int_4 NbPixels() const;
inline void setNbPixels(int_4 n) {nPix_= n;}
  
/* return the value of pixel number k */
virtual T& PixVal(int_4 k);
virtual T const& PixVal(int_4 k) const;

/* return the index of pixel at (theta,phi) */
virtual int_4 PixIndexSph(float theta, float phi) const;
   
/* return the spherical coordinates of center of pixel number k */
virtual void PixThetaPhi(int_4 k, float& theta, float& phi) const;

/* return the Pixel Solid angle  (steradians) */
virtual r_8 PixSolAngle(int_4 k) const; 

// ---------- Specific methods ------------------------------

void ReSize(int_4 nx, int_4 ny);

inline virtual char* TypeOfMap() const {return "LOCAL";};
 
/* Origin (with angle between x axis and phi axis, in degrees)  x0,y0  the default: middle of map*/
virtual void SetOrigin(float theta=90., float phi=0., float angle=0.);
virtual void SetOrigin(float theta,float phi,int_4 x0,int_4 y0,float angle=0.);

/* Pixel size (degres) */ 
virtual void SetSize(float angleX, float angleY);

/* Check to see if the local mapping is done */
inline bool LocalMap_isDone() const {return(originFlag_ && extensFlag_);};

/* Projection to/from spherical map */
virtual void Project(SphericalMap<T>& sphere) const;
  
/* There should be a more complex algorithm somewhere to combine *several* local maps to a full sphere.
      -> static method, or separate class */
  
/* provides a integer characterizing the pixelization refinement  (here : number of pixels) */
inline virtual int_4 SizeIndex() const {return(nPix_);}
inline int_4 Size_x() const {return nSzX_;}
inline void setSize_x(int_4 n) {nSzX_= n;}
inline int_4 Size_y() const {return nSzY_;}
inline void setSize_y(int_4 n) {nSzY_= n;}

inline void Origin(float& theta, float& phi,int& x0, int& y0, float& angle) const {theta= (float)theta0_; phi= (float)phi0_; x0= x0_; y0= y0_;angle= (float)angle_;}

inline void Aperture(float& anglex, float& angley) const {anglex= (float)angleX_; angley= (float)angleY_;}

/* retourne le pointeur vers/remplit  le vecteur des contenus des pixels */ 
inline const NDataBlock<T>* getDataBlock() const {return (&pixels_);}
inline void setDataBlock(T* data, int_4 n) {pixels_.FillFrom(n,data);}

/* impression */ 
void print(ostream& os) const;

// ---------- Mthodes internes -----------------------------
          
private :

void InitNul();
void Getij(int k, int& i, int& j) const;
void ReferenceToUser(float &theta, float &phi) const; 
void UserToReference(float &theta, float &phi) const;
void PixProjToAngle(float x, float y,float &theta, float &phi) const; 
void AngleProjToPix(float theta, float phi, float& x, float& y) const; 

// ---------- Variables internes ----------------------------

int_4 nSzX_;
int_4 nSzY_;
int_4 nPix_;
bool originFlag_;
bool extensFlag_;
int_4 x0_;
int_4 y0_;
r_8 theta0_;
r_8 phi0_;
r_8 angle_;
r_4 cos_angle_;
r_4 sin_angle_;
r_8 angleX_;
r_8 angleY_;
r_8 tgAngleX_;
r_8 tgAngleY_;
NDataBlock<T> pixels_;
};

// ------------- Classe pour la gestion de persistance --
template <class T>
class FIO_LocalMap : public PPersist  
{

public:

FIO_LocalMap();
FIO_LocalMap(string const & filename); 
FIO_LocalMap(const LocalMap<T>& obj);
FIO_LocalMap(LocalMap<T>* obj);
virtual ~FIO_LocalMap();
virtual AnyDataObj* DataObj();
inline operator LocalMap<T>() { return(*dobj); }
inline LocalMap<T> getObj() { return(*dobj); }

protected :

virtual void ReadSelf(PInPersist&);           
virtual void WriteSelf(POutPersist&) const;  
LocalMap<T>* dobj;
bool ownobj;
};

#endif
