// Peida Interactive   -   PI        R. Ansari 97-99
// Traceur (Drawer) pour NTupleInterface  
// LAL (Orsay) / IN2P3-CNRS  DAPNIA/SPP (Saclay) / CEA

#include <stdio.h>
#include "pintuple.h"

//++
// Class	PINTuple
// Lib		PIext
// include	pintuple.h
//
//	Classe de traceur 2D (dans un plan)  partir des donnes
//	d'un objet implmentant l'interface *NTupleInterface*.
//	Les objets "PINTuple" peuvent tracer des signes (markers)
//	ventuellement avec des barres d'erreur et une tiquette
//	pour chaque point. Si un attribut de ligne, autre que
//	"PI_NotDefLineAtt" est spcifi, les points sont connects
//	par une ligne.
//--
//++
// Links	Parents
// PIDrawer
//--
//++
// Links	Voir aussi
// NTupleInterface
// PINTuple3D
//--

//++
// Titre	Constructeur
//--
//++
// PINTuple(NTupleInterface* nt, bool ad)
//	Constructeur. Si "ad == true", l'objet "nt" est dtruit par 
//	le destructeur de l'objet "PINTuple"
//	Note : "nt" doit tre cr par new
//--
 
/* --Methode-- */
PINTuple::PINTuple(NTupleInterface* nt, bool ad)
: PIDrawer()
{
  mNT = nt;
  mAdDO = ad; 
  SetStats(true);
  SelectXY(NULL, NULL);
  SelectWt(NULL, 1);
  SelectErrBar();
  SelectLabel(NULL);
}

PINTuple::~PINTuple()
{
  if (mAdDO && mNT)  delete mNT;
}

//++
// Titre	Mthodes
//--
//++
// void  SelectXY(const char* px, const char* py)
//	Choix des noms de colonnes X,Y dfinissant les coordonnes des points. 
//	Ces deux colonnes doivent tre spcifies pour obtenir un trac.
// void  SelectErrBar(const char* erbx=NULL, const char* erby=NULL)
//	Choix des noms de colonnes pour le trac des barres d'erreur.
// void  SelectWt(const char* pw=NULL, int nbins=10)
//	Choix du nom de colonne poids. Dans ce cas, la taille du signe
//	(marker) sera proportionnel  la valeur de cette colonne pour 
//	chaque point.
// void  SelectLabel(const char* plabel=NULL)
//	Choix du nom de colonne correspondant  l'tiquette.
//--

/* --Methode-- */
void  PINTuple::SelectXY(const char* px, const char* py)
{
string name;
if (mNT == NULL)  xK = yK = -1;
if (px == NULL) xK = -1;
else { name = px; xK = mNT->ColumnIndex(name); }
if (py == NULL) yK = -1;
else { name = py; yK = mNT->ColumnIndex(name); }
}

/* --Methode-- */
void  PINTuple::SelectWt(const char* pw, int nbins)
{
nWbins = (nbins > 0) ? nbins : 10;
if (pw == NULL) wK = -1;  
else { string name = pw;   wK = mNT->ColumnIndex(name);  }
 
if (wK >= 0) mNT->GetMinMax(wK, wMin, wMax);
else  { wMin = 0.; wMax = 1.; }
}

/* --Methode-- */
void  PINTuple::SelectLabel(const char* plabel)
{
if (plabel == NULL) lK = -1;
else {  string name = plabel;  lK = mNT->ColumnIndex(name);  }
}

/* --Methode-- */
void  PINTuple::SelectErrBar(const char* erbx, const char* erby)
{
string name;
if (mNT == NULL)  xebK = yebK = -1;
if (erbx == NULL) xebK = -1;
else { name = erbx;  xebK = mNT->ColumnIndex(name); }
if (erby == NULL) yebK = -1;
else { name = erby;  yebK = mNT->ColumnIndex(name); }
}


/* --Methode-- */
void PINTuple::UpdateLimits()
{
  if (!mNT) return;
  if (mNT->NbLines() <= 0)  return;
  if ( (xK < 0) || (yK < 0) )   return;

  // Commencer par trouver nos limites
  double dx, dy;
  double xmin, xmax, ymin, ymax;
  xmin = ymin = 9.e19;
  xmax = ymax = -9.e19;
  mNT->GetMinMax(xK, xmin, xmax);
  mNT->GetMinMax(yK, ymin, ymax);
    
  dx = 0.02*(xmax-xmin);
  dy = 0.02*(ymax-ymin);
  
  SetLimits(xmin-dx, xmax+dx, ymin-dy, ymax+dy);
  SetAxesFlags(kBoxAxes | kExtTicks | kLabels);
}


/* --Methode-- */
void PINTuple::Draw(PIGraphicUC* g, double xmin, double ymin, double xmax, double ymax)
{
double xp,yp,xer,yer,wp;
double xl,yl;
int nok;

if (!mNT) return;
if ( (xK < 0) || (yK < 0) )  return;
if (mLAtt == PI_NotDefLineAtt)  g->SelLine(PI_ThinLine);

//  Pour tracer des markers avec taille fonction de Wt (poids)
double dw = (wMax-wMin)/nWbins;
if (dw < 1.e-19) dw = 1.e19;
int msz,sz;

PIMarker mrk;
if (wK >= 0)  mrk = (mMrk != PI_NotDefMarker) ? mMrk : PI_CircleMarker;
else   mrk = (mMrk != PI_NotDefMarker) ? mMrk : PI_DotMarker;
msz = mMSz;
if (msz < 1) msz = 1;
g->SelMarker(msz, mrk);

nok = 0;  
xp = yp = xl = yl = 0;
for (int i=0; i<mNT->NbLines(); i++) {
  xl = xp;  yl = yp; 
  xp = mNT->GetCell(i, xK);
  yp = mNT->GetCell(i, yK);
  if ( (xp < xmin) || (xp > xmax) || (yp < ymin) || (yp > ymax) )  continue;
  if ( (i > 0) && (mLAtt != PI_NotDefLineAtt) )   // On relie les points ...
    g->DrawLine(xl, yl, xp, yp);
  nok++;
  if ( xebK >= 0 ) {
    xer = mNT->GetCell(i, xebK);
    if(xer>0.) g->DrawLine(xp-xer, yp, xp+xer, yp);
  }
  if ( yebK >= 0 ) {
    yer = mNT->GetCell(i, yebK);
    if(yer>0.) g->DrawLine(xp, yp-yer, xp, yp+yer);
  }
  if (wK >= 0) { // Taille de marker en fonction du poids
    wp = mNT->GetCell(i, wK);
    sz = (int)((wp-wMin)/dw);
    if (sz < 0) sz = 0;
    if (sz > nWbins)  sz = nWbins;
    sz += msz;
    if (sz < 2)  g->SelMarker(sz, PI_DotMarker);
    else g->SelMarker(sz, mrk);
  }
  // Trace du marker
  if ((wK >= 0)||(lK < 0)||(mMrk != PI_NotDefMarker))  g->DrawMarker(xp, yp); 
  // Trace eventuel du label
  if (lK >= 0) g->DrawString(xp, yp, mNT->GetCelltoString(i, lK).c_str());

}

if (stats) { // Trace de stats 
  char label[64];
  sprintf(label, "Nd= %d/ Ntot= %d", nok, mNT->NbLines());
  g->SelFontSz((YMax() - YMin())/25, mFAtt);  
  PIGrCoord a,d;
  double cH = (double)g->GetFontHeight(a,d);
  double cellHeight = cH * 1.4;
  double cellWidth =  (double)g->CalcStringWidth(label) * 1.1;
  g->DrawLine(XMax() - cellWidth, YMax(),
                       XMax() - cellWidth, YMax() - cellHeight);
  g->DrawLine(XMax() - cellWidth, YMax() - cellHeight,
                       XMax()            , YMax() - cellHeight);
  g->DrawString(XMax() - cellWidth*0.95, YMax() - cH*1.2, label);
}

return;
}

/* --Methode-- */
void PINTuple::AppendTextInfo(string& info, double xmin, double ymin, double xmax, double ymax)
{
if (!mNT) return;
if ( (xK < 0) || (yK < 0) )  return;

int ncnt = 0;
double xp,yp;
char buff[128];
sprintf(buff,"PINTuple: NLines= %d  NCol= %d \n", mNT->NbLines(),  mNT->NbColumns());
info += buff;
info += mNT->LineHeaderToString();
for (int i=0; i<mNT->NbLines(); i++) {
  xp = mNT->GetCell(i, xK);
  yp = mNT->GetCell(i, yK);
  if ( (xp < xmin) || (xp > xmax) || (yp < ymin) || (yp > ymax) )  continue;
  ncnt++;
  if (ncnt > 101) continue;
  info += mNT->LineToString(i);
  }
if (ncnt >= 101) info += " .... \n";
sprintf(buff," %d points inside selected region \n", ncnt);
info += buff;
// printf("PINTuple::AppendTextInfo()-DBG %g %g %g %g - %d\n", xmin, ymin, xmax, ymax, ncnt); 
return;
}
