#ifndef SPHERETHETAPHI_SEEN
#define SPHERETHETAPHI_SEEN

#include "sphericalmap.h"
#include "ndatablock.h"
#include "tvector.h"

#include "anydataobj.h"
#include "ppersist.h"

// ***************** Class SphereThetaPhi *****************************
/*!    sphere splitted with respect to theta, phi : each hemisphere is 
    splitted into (m-1) parallels (equator does not enter into account).
   This operation defines m slices, each of which is splitted into 
    equidistant meridians. This splitting is realized in such a way that 
    all pixels have the same area and are as square as possible.

    One begins with the hemisphere with positive z, starting from the pole
    toward the equator. The first pixel is the polar cap ; it is circular
    and centered on theta=0. 
*/
template <class T>
class SphereThetaPhi : public SphericalMap<T>, public AnyDataObj
{

public :

SphereThetaPhi();
/*!    m is the number of slices in theta on an hemisphere (the polar cap
   forms the first slice).
    pet is a dummy parameter at the moment. 
*/
SphereThetaPhi(int m);
SphereThetaPhi(const SphereThetaPhi<T>& s);
virtual ~SphereThetaPhi();

// ------------ Definition of PixelMap abstract methods -

/* retourne/fixe le nombre de pixels */ 
/*!    Return total number of pixels  */
virtual int NbPixels() const;
inline void setNbPixels(int nbpix) { NPix_= nbpix; }

/* retourne la valeur du pixel d'indice k */ 
/*!    Return value of pixel with index k */
virtual T&       PixVal(int k);
virtual T const& PixVal(int k) const;

/* Return true if teta,phi in map  */
virtual bool ContainsSph(double theta, double phi) const;
/* retourne l'indice du pixel a (theta,phi) */ 
/*    Return index of the pixel corresponding to direction (theta, phi). */
virtual int PixIndexSph(double theta, double phi) const;

/* retourne les coordonnees Spheriques du centre du pixel d'indice k */ 
/*!   Return (theta,phi) coordinates of middle of  pixel with  index k */
virtual void PixThetaPhi(int k, double& theta, double& phi) const;

/* retourne/fixe l'angle Solide de Pixel   (steradians) */ 
/*!   Pixel Solid angle  (steradians)

    All the pixels have the same solid angle. The dummy argument is
   for compatibility with eventual pixelizations which would not 
    fulfil this requirement.
*/
virtual double PixSolAngle(int dummy) const;
inline void setPixSolAngle(double omega) { Omega_= omega; }
 
/* retourne/fixe la valeur du parametre de decoupage m */ 
inline virtual int SizeIndex() const { return( NTheta_); }
inline void setSizeIndex(int nbindex) { NTheta_= nbindex; }

// ------------- Specific methods  ----------------------

/*!   re-pixelize the sphere */
virtual void Resize(int m);

inline virtual char* TypeOfMap() const {return "TETAFI";};

/* Valeurs de theta des paralleles et phi des meridiens limitant le pixel d'indice k */
/*   Return values of theta,phi which limit the pixel with  index k */
virtual void Limits(int k,double& th1,double& th2,double& phi1,double& phi2);

/* Nombre de tranches en theta */ 
/*!    Return number of theta-slices on the sphere */
int NbThetaSlices() const;

/* Nombre de pixels en phi de la tranche d'indice kt */ 
int NPhi(int kt) const;

/* Renvoie dans t1,t2 les valeurs respectives de theta min et theta max  */
/* de la tranche d'indice kt  */ 
/*!    Return  theta values which limit the slice kt */
void Theta(int kt, double& t1, double& t2);

/* Renvoie dans p1,p2 les valeurs phimin et phimax du pixel d'indice jp  */
/* dans la tranche d'indice kt  */ 
/*!   Return values of phi which limit the jp-th pixel of the kt-th slice */
void Phi(int kt, int jp, double& p1, double& p2);

/* Renvoie l'indice k du pixel d'indice jp dans la tranche d'indice kt   */
/*!   Return pixel index  with sequence index jp in the slice kt */
int Index(int kt, int jp) const;

/* Indice kt de la tranche et indice jp du pixel d'indice k  */ 
/*!    Return indices kt (theta) and jp (phi) of  pixel with index k */
void ThetaPhiIndex(int k,int& kt,int& jp);

/*!    achieve the splitting into pixels (m has the same signification 
    as for the constructor)

    Each theta-slice of the north hemisphere will be spitted starting f
    from  phi=0 ...

    South hemisphere is scanned in the same direction according to phi
    and from equator to the pole (the pixel following the last one of
    the slice closest to the equator with z>0, is the pixel with lowest 
    phi of the slice closest of the equator with z<0).
*/
void Pixelize(int); 

/*!   For a theta-slice with index 'index', return : 

   the corresponding "theta" 

    a vector containing the phi's of the pixels of the slice

    a vector containing the corresponding values of pixels 
*/
void GetThetaSlice(int index,double& theta,TVector<double>& phi,TVector<T>& value) const; 

/*retourne le tableau contenant le nombre de pixels en phi de chacune des bandes en theta */
inline const int* getmNPhi() const { return(NPhi_); }
void setmNPhi(int* array, int m);

/* retourne/remplit le tableau contenant le nombre/Deuxpi total des pixels contenus dans les bandes */
inline const int* getmTNphi() const { return(TNphi_); }
void setmTNphi(int* array, int m);

/* retourne/remplit le tableau contenant les valeurs limites de theta */
inline const double* getmTheta() const { return(Theta_); }
void setmTheta(double* array, int m);

/* retourne le pointeur vers/remplit  le vecteur des contenus des pixels */ 
inline const NDataBlock<T>* getDataBlock() const { return (&pixels_); }
void setDataBlock(T* data, int m);

/* impression */ 
void print(ostream& os) const;

private :

// ------------- mthodes internes ----------------------          
void InitNul();
void Clear();

// ------------- variables internes ---------------------
int NTheta_;
int NPix_;
double Omega_;
int* NPhi_;
int* TNphi_;
double* Theta_;
NDataBlock<T> pixels_;
};

// ------------- Classe pour la gestion de persistance --
template <class T>
class FIO_SphereThetaPhi : public PPersist  
{
public:

FIO_SphereThetaPhi();
FIO_SphereThetaPhi(string const & filename); 
FIO_SphereThetaPhi(const SphereThetaPhi<T>& obj);
FIO_SphereThetaPhi(SphereThetaPhi<T>* obj);
virtual ~FIO_SphereThetaPhi();
virtual AnyDataObj* DataObj();
inline operator SphereThetaPhi<T>() { return(*dobj); }
//inline SphereThetaPhi<T> getObj() { return(*dobj); }

protected :

virtual void ReadSelf(PInPersist&);           
virtual void WriteSelf(POutPersist&) const;  
SphereThetaPhi<T>* dobj;
bool ownobj;
};

#endif
