// toiiter.cc
// Eric Aubourg         CEA/DAPNIA/SPP   juillet 1999

#include "ark.h"

#include "toiiter.h"
#include "toimanager.h"
#include "toiproducer.h"
#include "archparam.h"
#include "asigps.h"
#include "toiauxgpsproducer.h"
#include <iostream.h>
#include <fstream.h>

#define CHK_INITED if (initDone) \
        throw ArchExc("Trying to modify TOIIter after init done.");

TOIIter::TOIIter() {
  mjdStart = -999999999;
  mjdEnd   =  999999999;
  utcStart = -999999999;
  utcEnd   =  999999999;
  sStart   = -999999999;
  sEnd     =  999999999;

  underSample = 1;
  curSample = -1;
  
  initDone = false;
  incDone  = false;
}

#ifdef __MWERKS__
#pragma mark -
#endif

void TOIIter::addDirectory(string dir) {
  CHK_INITED
  fset.addDirectory(dir);
}

void TOIIter::addFile(string fn) {
  CHK_INITED
  fset.addFile(fn);
}

void TOIIter::setMJDInterval(double tStart, double tEnd) {
  CHK_INITED
  if (tStart>0) mjdStart = tStart;
  if (tEnd>0)   mjdEnd = tEnd;
}

void TOIIter::setUTCInterval(double tStart, double tEnd) {
  CHK_INITED
  if (tStart>0) utcStart = tStart;
  if (tEnd>0)   utcEnd = tEnd;
}

void TOIIter::setSNInterval(long tStart, long tEnd) {
  CHK_INITED
  if (tStart>0) sStart = tStart;
  if (tEnd>0)   sEnd = tEnd;
}

void TOIIter::setUnderSample(int n) {
  CHK_INITED
  if (n<=1) n=1;
  underSample = n;
}

int TOIIter::getUnderSample() {
  return underSample;
}

void TOIIter::addTOI(TOI& toi, bool trg) {
  CHK_INITED
  TOIProducer* prod = TOIManager::findTOIProducer(toi);
  if (!prod) throw ArchExc("Cannot produce " + toi.fullName());
  prod->addTOI(toi, this);
  request.push_back(TOIInfo(toi,prod,trg?1:0));
}

#ifdef __MWERKS__
#pragma mark -
#endif


void TOIIter::registerReqHandler(RequestHandler* h) {
  CHK_INITED
  handlers.push_back(h);
}


void TOIIter::readReq(istream& str) {
  CHK_INITED
   if (!incDone) defaultInclude();
   string line;
   while (str) {
     getline(str,line);
     if (!str) break;
     if (line.substr(0,4)=="#END" && (line.length()==4 || line[5] == ' ')) break;
     if (line[0] != '@' && line[0] != '#') continue;
     if (!processRequest(line)) {
       throw ArchExc("Unrecognized directive " + line);
     }
   }
}


bool TOIIter::processRequest(string line) {
  if (line[0] == '#') {
    int x = line.find(' ');
    string keyw = line.substr(0, x);
    string args = (x>0) ? line.substr(x) : string("");
    bool handled = processOption(keyw,args);
    for (list<RequestHandler*>::iterator i = handlers.begin(); 
	 i != handlers.end(); i++) {
      handled |= (*i)->processOption(keyw,args);
    }
    return handled;
  }

  if (line[0] == '@') {
    TOI toi(line.substr(1));
    // if (kind  == sampleNum || kind == mjd || kind == mutc) notrig = true;
    bool handled = processTOIReq(toi,line);
    for (list<RequestHandler*>::iterator i = handlers.begin(); 
         i != handlers.end(); i++) {
      handled |= (*i)->processTOIReq(toi,line);
    }
    return handled;
  }
  return false;
}


bool TOIIter::processTOIReq(TOI& toi,string)
{
  TOI toi2 = toi;
  bool trg=true;
  if (toi.options.find("notrig") != toi.options.end()) {
    toi2.options.erase("notrig");
    trg=false;
  } 
  addTOI(toi2, trg);
  toi.unit = toi2.unit;
  return true;
} 


bool TOIIter::processOption(string key, string arg)
{
  if (arg.length()>0 && arg[0] == ' ') {
    arg = arg.substr(arg.find_first_not_of(' '));
  }
  if (key == "#MJDRANGE") {
    double tmin, tmax;
    sscanf(arg.c_str(), "%lg %lg", &tmin, &tmax);
    setMJDInterval(tmin, tmax);
  } else if (key == "#UTCRANGE") {
    double tmin, tmax;
    sscanf(arg.c_str(), "%lg %lg", &tmin, &tmax);
    setUTCInterval(tmin, tmax);
  } else if (key == "#SNRANGE") {
    long tmin, tmax;
    sscanf(arg.c_str(), "%ld %ld", &tmin, &tmax);
    setSNInterval(tmin, tmax);
  } else if (key == "#PATH") {
    addDirectory(arg);
  } else if (key == "#FILE") {
    addFile(arg);
  } else if (key == "#UNDERSAMPLE") {
    setUnderSample(atoi(arg.c_str()));
  } else if (key == "#MJD0") {
    double t0;
    sscanf(arg.c_str(), "%lg", &t0);
    archParam.acq.tBlock0 = t0;
  } else if (key == "#UTCORIGIN") {
    double t0;
    sscanf(arg.c_str(), "%lg", &t0);
    archParam.acq.utcOrigin = t0;
  } else if (key == "#PERECH") {
    double t0;
    sscanf(arg.c_str(), "%lg", &t0);
    archParam.acq.perEch = t0;
  } else if (key == "#ASIGPS") {
    ASIGPS* gps = new ASIGPS(arg);
    TOIManager::registerProducer(new TOIAuxGPSProducer(gps));
    //    gps->FitsDump("GPSDump.fits");
  } else if (key == "#INCLUDE") {
    ifstream f(arg.c_str());
    readReq(f);
  } else {
    return false;
  }
  return true;
}


void TOIIter::defaultInclude() {
  incDone = true;
  processRequest("#REQVERSION V_270999");
  processRequest("#MJD0 1376.8358818");
  processRequest("#PERECH 0.005836818076");
  processRequest("#UTCORIGIN 1376.5");
  processRequest("#ASIGPS ASI_GPS_archeops1999.ascii");
  processRequest("#COMMENT Archtoi V3 -- novembre 1999 -- Eric Aubourg CEA/DAPNIA");
}

#ifdef __MWERKS__
#pragma mark -
#endif

void TOIIter::init() {
  if (initDone) return;
  initDone = true;
 
  if (utcStart > 0) {
    double t = (utcStart/24.) + archParam.acq.utcOrigin;
    if (t > mjdStart) mjdStart=t;
  }
  if (utcEnd > 0) {
    double t = (utcEnd/24.) + archParam.acq.utcOrigin;
    if (t < mjdEnd) mjdEnd=t;
  }

  // Let's add some time on each side, 30 seconds should be ok, at least
  // for Trapani
  
  double delT = 30. / 86400;
  long   delSN = long(delT / archParam.acq.perEch);

  fset.setMJDRange(mjdStart-delT, mjdEnd+delT);
  fset.setSNumRange(sStart-delSN, sEnd+delSN);

  fset.init();
  //curSample = fset.getSampleIndex();
}

bool TOIIter::isTrig(int column) {
  return (request[column].third & triggering) != 0;
}
 
TOI TOIIter::getKind(int column) {
  return request[column].first;
}

long TOIIter::getSampleNum() {
  return curSample;
}

bool TOIIter::next() {
  if (!initDone) init();
  for (int ii=0; ii<underSample; ii++)
    if (!next1()) return false;
  
  return true;
}

bool TOIIter::next1() {
// On tente de produire curSample+1 pour toutes les toi
// Eventuellement en avancant sur le fichier...
// Puis on regarde si une TOI triggering a eu une nouvelle
// valeur.
// Si on a epuise les fichiers de donnees, on s'arrete des qu'aucune
// TOI n'a de valeurs apres curSample...
  if (curSample <= 0) curSample = sStart-1;  
  if (curSample <= 0) {
    long sn = archParam.acq.MJD2SN(mjdStart) - 1;
    if (sn > curSample) curSample = sn;
  }
  
  if (curSample <= 0) curSample = fset.getSampleIndex()-1;
  
  if (curSample >= sEnd || archParam.acq.SN2MJD(curSample) >= mjdEnd) return false;
  
  static long lastClean = 0;
  if (curSample - lastClean > 100) {
    for (int i=0; i<request.size(); i++)
      request[i].second->wontNeedEarlier(request[i].first, this, curSample);
    lastClean = curSample;
  }

  bool endFound = false;
  while(1) {
    curSample++;
    for (vector<TOIInfo>::iterator i = request.begin(); i != request.end(); i++) {
      while ((*i).second->canGetValueLater(curSample, (*i).first)) {
        if (! fset.next()) {endFound = true; break;} //return false;
      }     
    }
    bool found=false;
    //bool valuesAhead = false;
    for (vector<TOIInfo>::iterator i = request.begin(); i != request.end(); i++) {
      if (((*i).third & triggering) == 0) continue;
      if ((*i).second->canGetValue(curSample, (*i).first)) {found=true;break;}
      //if ((*i).second->lastSampleNum((*i).first)>curSample) valuesAhead=true;
    }
    if (found) break;
    if (endFound && curSample >= fset.getSampleIndex()+71) return false;
  }
  return true;
}

int TOIIter::getColTOI(TOI const& toi) {
  for (int i=0; i<request.size(); i++)
    if (request[i].first == toi) return i;
  throw ArchExc("getColTOI : no such TOI " + toi.name);
}

bool TOIIter::canGetValue(int column) {
   TOIInfo& info = request[column];
   return(info.second->canGetValue(curSample, info.first));
}

double TOIIter::getValue(int column) {
   TOIInfo& info = request[column];
   return(info.second->getValue(curSample, info.first));
}

bool TOIIter::canGetValue(TOI const& toi) {
  return canGetValue(getColTOI(toi));
}

double TOIIter::getValue(TOI const& toi) {
  return getValue(getColTOI(toi));
}

