// This may look like C code, but it is really -*- C++ -*-
//      Base array class - Memory organisation management
//                     R. Ansari, C.Magneville   03/2000

#ifndef BaseArray_SEEN
#define BaseArray_SEEN

#include "machdefs.h"
#include <math.h>
#include <iostream.h>
#include "anydataobj.h"
#include "dvlist.h"


//  Maximum number of dimensions for array
#define BASEARRAY_MAXNDIMS  5

namespace SOPHYA {  

//   ------------ classe template Array ----------- 
class BaseArray : public AnyDataObj {
public:
  // Creation / destruction 
  BaseArray();
  virtual ~BaseArray();

  // Returns true if ndim and sizes are equal
  virtual bool CompareSizes(const BaseArray& a);

  // Compacts size=1 array dimensions
  virtual void CompactAllDim();         // suppresses all size==1 dimensions 
  virtual void CompactTrailingDim();    // suppresses size==1 dimensions after the last size>1 dimension

  // Array dimensions
  inline uint_4 NbDimensions() const { return( ndim_ ); }

  inline uint_8 Size() const { return(totsize_); }
  inline uint_4 SizeX() const { return(size_[0]); }
  inline uint_4 SizeY() const { return(size_[1]); }
  inline uint_4 SizeZ() const { return(size_[2]); }
  inline uint_4 Size(int ka) const { return(size_[CheckDI(ka,1)]); }

  uint_4 MaxSizeKA() const ;

  // memory organisation - packing information
  inline bool   IsPacked() const { return(moystep_ == 1); }
  inline bool   IsPackedX() const { return(step_[0] == 1); }
  inline bool   IsPackedY() const { return(step_[1] == 1); }
  inline bool   IsPackedZ() const { return(step_[2] == 1); }
  inline bool   IsPacked(int ka) const { return(step_[CheckDI(ka,2)] == 1); }

  inline uint_4 MinStep() const  { return(minstep_); }
  inline uint_4 AvgStep() const  { return(moystep_); }
  inline uint_4 StepX() const { return(step_[0]); }
  inline uint_4 StepY() const { return(step_[1]); }
  inline uint_4 StepZ() const { return(step_[2]); }
  inline uint_4 Step(int ka) const { return(step_[CheckDI(ka,3)]); }

  uint_4 MinStepKA() const ;

  uint_8 Offset(uint_8 ip=0) const ;
  inline uint_8  Offset(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it=0, uint_4 iu=0) const;  

//  a abstract element acces methode
  virtual double ValueAtPosition(uint_8 ip) const = 0;

// Impression, I/O, ...
  static void   SetMaxPrint(uint_4 nprt=50);   
  void          Show(ostream& os, bool si=false) const;
  inline void   Show() const { Show(cout); }
  virtual string DataType() const = 0;

//  Objet DVList info 
  DVList&       Info();

protected:
  // Verifie la compatibilite de l'index de dimension
  inline int CheckDI(int ka, int msg) const ;
  // Verifie la compatibilite des bornes d'index
  inline void CheckBound(int ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu, int msg) const ;
  // Changing Sizes/NDim ... return true if OK
  bool UpdateSizes(uint_4 ndim, const uint_4 * siz, uint_4 step, uint_8 offset, string & exmsg);
  bool UpdateSizes(uint_4 ndim, const uint_4 * siz, const uint_4 * step, uint_8 offset, string & exmsg);
  bool UpdateSizes(const BaseArray& a, string & exmsg);
  static uint_8 ComputeTotalSize(uint_4 ndim, uint_4 * siz, uint_4 step, uint_8 offset) ;
  // Pour Extraction de sous-tableau
  virtual void SubArray(BaseArray & ra, uint_4 ndim, uint_4 * siz, uint_4 * pos, uint_4 * step);

  uint_4 ndim_;                   // nb of dimensions
  uint_4 size_[BASEARRAY_MAXNDIMS];     // array size in each dimension
  uint_8 totsize_;                   // Total number of elements
  uint_4 step_[BASEARRAY_MAXNDIMS];     // two consecutive elements distance in a given dimension 
  uint_4 minstep_;                   // minimal step (in any axes)
  uint_4 moystep_;                   // mean step 0 non regular steps
  uint_8 offset_;              // global offset -> position of elem[0] in DataBlock
  uint_4 marowi_, macoli_;     // For matrices, Row index and column index in dimensions

  DVList* mInfo;               // Infos (variables) attachees au tableau

  static char * ck_op_msg_[6];  // Operation messages for CheckDI() CheckBound()
  static uint_4 max_nprt_;      // Nb maxi d'elements imprimes
};

// --------------------------------------------------
//        Methodes inline de verification
// --------------------------------------------------
inline int BaseArray::CheckDI(int ka, int msg)  const
{
  if ( (ka < 0) || (ka >= ndim_) ) {
    string txt = "BaseArray::CheckDimensionIndex/Error ";   txt += ck_op_msg_[msg];
    throw(ParmError(txt));
  }
  return(ka);
}

inline void BaseArray::CheckBound(int ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu, int msg)  const
{
  if ( (ix >= size_[0]) || (iy >= size_[1]) || (iz > size_[2]) ||
       (it >= size_[3]) || (iu >= size_[4]) ) {
    string txt = "BaseArray::CheckArrayBound/Error ";   txt += ck_op_msg_[msg];
    throw(ParmError(txt));
  }
  return;
}


// --------------------------------------------------
//        Position d'un element
// --------------------------------------------------
inline uint_8 BaseArray::Offset(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu) const
{
#ifdef SO_BOUNDCHECKING
  CheckBound(ix, iy, iz, it, iu, 4);
#endif
  return ( offset_+  ix*step_[0] + iy*step_[1] + iz*step_[2] +
	      it*step_[3] + iu*step_[4] );
}


} // Fin du namespace

#endif
