#include <stdio.h>
#include "pihisto2d.h"

#include PIMENU_H

PIHisto2D::PIHisto2D(Histo2D* histo, bool ad)
: PIScDrawer(), mHisto(histo)
{
  mAdDO = ad;     // Flag pour suppression automatique de mHisto  
  UseColors();
}

PIHisto2D::~PIHisto2D()
{
  if (mAdDO)
    delete mHisto;
}

void
PIHisto2D::UseColors(bool fg, CMapId cmap)
{
  mFgCol = fg; mCmap = cmap;
}

void
PIHisto2D::UpdateSize()
{
  if (!mDrawer) return;
  if (mDrawer->LimitsFixed()) return;
  if (!mHisto)  return;
  // Commencer par trouver nos limites
  
  float xmin = mHisto->XMin();
  float xmax = mHisto->XMax();

  float ymax = mHisto->YMax();
  float ymin = mHisto->YMin();
  
  if (mDrawWdg) 
    mDrawWdg->SetLimits(xmin, xmax, ymin, ymax);
  else 
    mDrawer->SetLimits(xmin, xmax, ymin, ymax);

  mDrawer->SetAxesFlags(kBoxAxes | kExtTicks | kLabels);
}


void
PIHisto2D::Draw(PIGraphicUC* g, float /*xmin*/, float /*ymin*/, float /*xmax*/, float /*ymax*/)
{
  if (!mHisto)  return;
  float hmin = mHisto->VMin();
  float hmax = mHisto->VMax();
  if(hmin>=hmax) hmax = hmin+1.;
  float dx = mHisto->WBinX();
  float dy = mHisto->WBinY();
  float fracmin=0.1, fracmax=0.8;

  PIColorMap* cmap=NULL;
  PIColors coul;
  int ncol = 0;
  if (mFgCol) { 
    cmap = new PIColorMap(mCmap);
    ncol = cmap->NCol();
    coul = g->GetForeground();
//    fracmin = 0.;
//    fracmax = 1.;
  }

  for (int i=0; i<mHisto->NBinX(); i++) {
    for (int j=0; j<mHisto->NBinY(); j++) {
      float left,bottom;
      mHisto->BinLowEdge(i,j,left,bottom);
      float frac = fracmax*((*mHisto)(i,j)-hmin)/(hmax-hmin);
      if(frac<fracmin) continue;
      float width = frac*dx;
      float height = frac*dy;
      left += 0.5*(1.-frac)*dx;
      bottom += 0.5*(1.-frac)*dy;
      if (cmap) {
        int icol = ncol*(frac/fracmax);
        if(icol>=ncol) icol = ncol-1; else if(icol<0) icol=0;
        g->SelForeground(*cmap,icol);
        g->DrawFBox(left,bottom,width,height);
      }
      else g->DrawBox(left,bottom,width,height);
    }
  }
  if (cmap) { 
    g->SelForeground(coul);
    delete cmap;
  }
  DrawStats(g);
}

void
PIHisto2D::DrawStats(PIGraphicUC* g)
{
  if (!mDrawWdg) return;
  // Une boite dans le coin superieur droit
  float cellHeight = (mDrawWdg->YMax() - mDrawWdg->YMin()) * 0.05;
  float cellWidth  = (mDrawWdg->XMax() - mDrawWdg->XMin()) * 0.23;
  g->DrawLine(mDrawWdg->XMax() - cellWidth, mDrawWdg->YMax(),
                       mDrawWdg->XMax() - cellWidth, mDrawWdg->YMax() - cellHeight);
  g->DrawLine(mDrawWdg->XMax() - cellWidth, mDrawWdg->YMax() - cellHeight,
                       mDrawWdg->XMax()            , mDrawWdg->YMax() - cellHeight);
  char label[50];
  sprintf(label, "N = %.6g", mHisto->NData());
  g->SelFontSz((mDrawWdg->YMax() - mDrawWdg->YMin())/30);
  g->DrawString(mDrawWdg->XMax() - cellWidth*0.9, mDrawWdg->YMax() - cellHeight*0.8, label);
                       
}


/////////////////////////////////////////////////////////////////
//  Classe PIH2DWdg
/////////////////////////////////////////////////////////////////


//  Le menu qui va servir a choisir l'option d'affichage suite au click 
// de bouton-3
static PIPUMenu* optmenh2d=NULL;
static int nb_optmen = 0;

PIH2DWdg::PIH2DWdg(PIContainerGen *par, char *nom, int sx, int sy, int px, int py)
        : PIScDrawWdg(par,nom,sx,sy,px,py)
{
if (!optmenh2d) {
  optmenh2d = new PIPUMenu((PIMsgHandler *)this, "TypeH2D");
  optmenh2d->AppendItem("B&W", 6101);
  optmenh2d->AppendItem("Gris", 6102);
  optmenh2d->AppendItem("RJ32", 6103);
  optmenh2d->AppendItem("BR32", 6104);
  }
nb_optmen++;
mPih = NULL;
ActivateButton(3); // Pour afficher le menu option de trace
}

PIH2DWdg::~PIH2DWdg()
{
nb_optmen--;
if (nb_optmen == 0) { delete optmenh2d;  optmenh2d=NULL; }
if (mPih)  delete mPih;
}

void PIH2DWdg::SetHisto(Histo2D* histo, bool fg, CMapId cmap)
{
if (!histo) return;
if (mPih)  delete mPih;
mPih = new PIHisto2D(histo, true);
mPih->UseColors(fg, cmap);
AddScDrawer(mPih);
}

string  PIH2DWdg::GetClickText(float x, float y)
{
int i,j;
char str[128];

if ((!mPih) || (!mPih->Histogram())) {
  sprintf(str,"X=%g Y=%g ???",x,y);
  return((string)str); 
}

Histo2D* h = mPih->Histogram();


h->FindBin(x,y,i,j);
if(i<0 || i>=h->NBinX() || j<0 || j>=h->NBinY())
  sprintf(str,"X=%g Y=%g ???",x,y);
else sprintf(str,"X=%g Y=%g v=%g",x,y,(*h)(i,j));

return((string)str);
}

void PIH2DWdg::Process(PIMessage msg, PIMsgHandler* sender, void* data)
{
//printf("PIH2DWdg::Process(%d-%d , %lx ...) \n", UserMsg(msg), ModMsg(msg), (long)sender);
if (!mPih) return;
if (sender == optmenh2d) { // On n'attend de message que de la part du menu 
  int opt=6101;
  bool fgc=false;
  CMapId col=CMAP_GREYINV32;
      
  opt = UserMsg(msg);
  if (opt == 6102) { fgc = true; col = CMAP_GREYINV32; }
  else if (opt == 6103) { fgc = true;  col = CMAP_COLRJ32; }
  else if (opt == 6104) { fgc = true; col = CMAP_COLBR32; }
  mPih->UseColors(fgc, col);
  mPih->Refresh();  // On rafraichit le dessin     
}

}

void PIH2DWdg::But3Press(int x, int y)
{
//printf("PIH2DWdg::But3Press(%d %d ) \n", x, y);
optmenh2d->SetMsgParent((PIMsgHandler*)this);
optmenh2d->Show(this, x, y);
}
