#include "piscdrawwdg.h"
#include "generalfit.h"
#include <algo.h>


PIScDrawWdg::PIScDrawWdg(PIContainerGen *par, char *nom,
                         int sx, int sy, int px, int py)
:PIBaseWdg(par, nom, sx, sy, px, py)
{
  SetLimits(-1, 1, -1, 1);
  limitsFixed = 0;
  xEndDrag = yEndDrag = xBegDrag = yBegDrag = 0;
}

PIScDrawWdg::~PIScDrawWdg()
{
  DeleteDrawers();
}


void
PIScDrawWdg::SetLimits(double xmin, double xmax, 
                       double ymin, double ymax, bool tmp)
{
  if (xmax <= xmin || ymax <= ymin)
    return; // $CHECK$ exception ?
  xMin = xmin;
  xMax = xmax;
  yMin = ymin;
  yMax = ymax;
  if (!tmp) {
    xMinS = xmin;
    xMaxS = xmax;
    yMinS = ymin;
    yMaxS = ymax;
  }
  CalcScale();
  CalcTicks();
  limitsFixed = 1;
}

void
PIScDrawWdg::Resize()
{
  CalcScale();
  CalcTicks();
  Refresh();
}

void
PIScDrawWdg::CalcScale()
{
  xScale = XSize() / (xMax-xMin);
  yScale = -YSize() / (yMax-yMin);
  xOrg   = -xMin * xScale + XPos();
  yOrg   = -yMax * yScale + YPos();
}

void
PIScDrawWdg::CalcTicks()
{
	double xExt = xMax-xMin;
	int lx = (int) ceil(log10(xExt)-0.1);
	xExt = pow(10.0,lx);
	xMajTickStep = xExt/10;
	xMinTickStep = xMajTickStep/10;
    xFirstMajTick = int(xMin / xMajTickStep) * xMajTickStep;
    if (xFirstMajTick < xMin) xFirstMajTick += xMajTickStep;
    xFirstMinTick = int(xMin / xMinTickStep) * xMinTickStep;
    if (xFirstMinTick < xMin) xFirstMinTick += xMinTickStep;
    
    
	double yExt = yMax-yMin;
	int ly = (int) ceil(log10(yExt)-0.1);
	yExt = pow(10.0,ly);
	yMajTickStep = yExt/10;
	yMinTickStep = yMajTickStep/10;
    yFirstMajTick = int(yMin / yMajTickStep) * yMajTickStep;
    if (yFirstMajTick < yMin) yFirstMajTick += yMajTickStep;
    yFirstMinTick = int(yMin / yMinTickStep) * yMinTickStep;
    if (yFirstMinTick < yMin) yFirstMinTick += yMinTickStep;
    
    xMajTickLen = -2/yScale;
    xMinTickLen = -1/yScale;
    yMajTickLen = 2/xScale;
    yMinTickLen = 1/xScale;
}

void
PIScDrawWdg::Sc2Local(double x, double y, int& xpix, int& ypix)
{
  xpix = int(xOrg + x*xScale + .5);
  ypix = int(yOrg + y*yScale + .5);
}

void
PIScDrawWdg::Local2Sc(int xpix, int ypix, double& x, double& y)
{
  x = (xpix-xOrg)/xScale;
  y = (ypix-yOrg)/yScale;
}


void
PIScDrawWdg::DSc2Local(double dx, double dy, int& dxpix, int& dypix)
{
  dxpix = int(dx*xScale + .5);
  dypix = int(dy*yScale + .5);
}

void
PIScDrawWdg::ScDrawString(double x, double y, char* s, int pos)
{
  int xPix, yPix;
  Sc2Local(x, y, xPix, yPix);
  DrawString(xPix, yPix, s, pos);
}

void
PIScDrawWdg::ScDrawLine(double x1, double y1, double x2, double y2)
{
  int x1Pix, y1Pix, x2Pix, y2Pix;
  Sc2Local(x1, y1, x1Pix, y1Pix);
  Sc2Local(x2, y2, x2Pix, y2Pix);
  DrawLine(x1Pix, y1Pix, x2Pix, y2Pix);
}  

void
PIScDrawWdg::ScDrawBox(double x, double y, double dx, double dy)
{
  int xPix, yPix, x2Pix, y2Pix;
  Sc2Local(x, y, xPix, yPix);
  Sc2Local(x+dx, y+dy, x2Pix, y2Pix);
  DrawBox(xPix, yPix, x2Pix-xPix, y2Pix-yPix);
}  

void
PIScDrawWdg::ScDrawFBox(double x, double y, double dx, double dy)
{
  int xPix, yPix, x2Pix, y2Pix;
  Sc2Local(x, y, xPix, yPix);
  Sc2Local(x+dx, y+dy, x2Pix, y2Pix);
  DrawFBox(xPix, yPix, x2Pix-xPix, y2Pix-yPix);
}  

void
PIScDrawWdg::ScDrawCircle(double x, double y, double r)
{
  int xPix, yPix, rPix;
  Sc2Local(x, y, xPix, yPix);
  rPix = r * xScale;
  DrawCircle(xPix, yPix, rPix);
}  

void
PIScDrawWdg::ScDrawFCircle(double x, double y, double r)
{
  int xPix, yPix, rPix;
  Sc2Local(x, y, xPix, yPix);
  rPix = r * xScale;
  DrawFCircle(xPix, yPix, rPix);
}  

void
PIScDrawWdg::DrawAxes(int tickLevel, int markLevel)
{
  // Les axes
  
  ScDrawLine(xMin, 0, xMax, 0);
  ScDrawLine(0, yMin, 0, yMax);
  
  // Les ticks majeurs
  
  if (tickLevel > 0) {
    for (double x=xFirstMajTick; x<=xMax; x += xMajTickStep)
      ScDrawLine(x, -xMajTickLen, x, xMajTickLen);
    for (double y=yFirstMajTick; y<=yMax; y += yMajTickStep)
      ScDrawLine(-yMajTickLen, y, yMajTickLen, y);
  }

  // Les ticks mineurs
  
  if (tickLevel > 1) {
    for (double x=xFirstMinTick; x<=xMax; x += xMinTickStep)
      ScDrawLine(x, -xMinTickLen, x, xMinTickLen);
    for (double y=yFirstMinTick; y<=yMax; y += yMinTickStep)
      ScDrawLine(-yMinTickLen, y, yMinTickLen, y);
  }


}

void
PIScDrawWdg::SetBBoxMargin(double h, double v)
{
  bbmargeh = h;
  bbmargev = v; 
}

void
PIScDrawWdg::DrawBBox(int tickLevel, int tickFlags, int markLevel)
{

  double xLow = xMin + bbmargeh;
  double xHig = xMax - bbmargeh;
  double yLow = yMin + bbmargev;
  double yHig = yMax - bbmargev;
  
  double extXMajTickLen = tickFlags&2 ? xMajTickLen : 0;
  double intXMajTickLen = tickFlags&1 ? xMajTickLen : 0;
  double extXMinTickLen = tickFlags&2 ? xMinTickLen : 0;
  double intXMinTickLen = tickFlags&1 ? xMinTickLen : 0;
  double extYMajTickLen = tickFlags&2 ? yMajTickLen : 0;
  double intYMajTickLen = tickFlags&1 ? yMajTickLen : 0;
  double extYMinTickLen = tickFlags&2 ? yMinTickLen : 0;
  double intYMinTickLen = tickFlags&1 ? yMinTickLen : 0;
  
  
  // Les bordures
  
  ScDrawLine(xLow, yLow, xHig, yLow);
  ScDrawLine(xHig, yLow, xHig, yHig);
  ScDrawLine(xHig, yHig, xLow, yHig);
  ScDrawLine(xLow, yHig, xLow, yLow);
  
  // Les ticks majeurs
  
  if (tickLevel > 0) {
    for (double x=xFirstMajTick; x<=xHig; x += xMajTickStep) {
      ScDrawLine(x, yLow - extXMajTickLen, x, yLow + intXMajTickLen);
      ScDrawLine(x, yHig + extXMajTickLen, x, yHig + intXMajTickLen);
    }
    for (double y=yFirstMajTick; y<=yHig; y += yMajTickStep) {
      ScDrawLine(xLow - extYMajTickLen, y, xLow + intYMajTickLen, y);
      ScDrawLine(xHig + extYMajTickLen, y, xHig - intYMajTickLen, y);
    }
  }
  

  // Les ticks mineurs
  
  if (tickLevel > 1) {
    for (double x=xFirstMinTick; x<=xHig; x += xMinTickStep) {
      ScDrawLine(x, yLow - extXMinTickLen, x, yLow + intXMinTickLen);
      ScDrawLine(x, yHig + extXMinTickLen, x, yHig + intXMinTickLen);
    }
    for (double y=yFirstMinTick; y<=yHig; y += yMinTickStep) {
      ScDrawLine(xLow - extYMinTickLen, y, xLow + intYMinTickLen, y);
      ScDrawLine(xHig + extYMinTickLen, y, xHig - intYMinTickLen, y);
    }
  }
  
  // Les labels
  if (markLevel > 0) {
    for (double x=xFirstMajTick; x<=xHig; x += xMajTickStep) {
      char label[20]; sprintf(label, "%-6g", x);
      ScDrawString(x, yLow - 8 * xMajTickLen, label);
      ScDrawString(x, yHig + 2 * xMajTickLen, label);
    }
    for (double y=yFirstMajTick; y<=yHig; y += yMajTickStep) {
      char label[20]; sprintf(label, "%-6g", y);
      ScDrawString(xLow - 8 * yMajTickLen, y, label);
      ScDrawString(xHig + 2 * yMajTickLen, y, label);
    }
  }
}

void
PIScDrawWdg::AddDrawer(PIScDrawer* d)
{
  mDrawers.push_back(d);
  d->AttachTo(this);
}

void
PIScDrawWdg::RemoveDrawer(PIScDrawer* d)
{
  mDrawers.remove(d);
  d->mDrawWdg = NULL;
}

void
PIScDrawWdg::DeleteDrawers()
{
  for(list<PIScDrawer*>::iterator i = mDrawers.begin(); i != mDrawers.end(); i++)
    delete *i;
    
  mDrawers.erase(mDrawers.begin(),mDrawers.end());
}

void
PIScDrawWdg::CallDrawer(PIScDrawer* d)
{
  d->Draw();
}

void
PIScDrawWdg::CallDrawers()
{
  for_each(mDrawers.begin(), mDrawers.end(), CallDrawer);
}

void
PIScDrawWdg::SetAxesKind(int axesKind, int tickLevel, int markLevel)
{
  axesChoice  = axesKind;
  tickChoice  = tickLevel;
  labelChoice = markLevel;
}

void
PIScDrawWdg::Draw()
{
  EraseWindow();
  if (axesChoice == 1)
    DrawAxes(tickChoice, labelChoice);
  else if (axesChoice == 2)
    DrawBBox(tickChoice, labelChoice);
  CallDrawers();
}


// Gestion du zoom

void
PIScDrawWdg::But2Press(int x, int y)
{
  SelPointerShape(PI_CrossPointer);
  xEndDrag = xBegDrag = x;
  yEndDrag = yBegDrag = y;
  PIBeep();
}

void
PIScDrawWdg::But2Release(int x, int y)
{

  SelPointerShape(PI_ArrowPointer);
  SelForeground(PI_White);
  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);
  SelForeground(PI_Black);
  
  if (xBegDrag == xEndDrag || yBegDrag == yEndDrag)
    SetLimits(xMinS, xMaxS, yMinS, yMaxS);
  else {
    double xl,yl,xh,yh;
    Local2Sc(xBegDrag, yBegDrag, xl, yl);
    Local2Sc(xEndDrag, yEndDrag, xh, yh);
    if (xl > xh) swap(xl, xh);
    if (yl > yh) swap(yl, yh);
    SetLimits(xl,xh,yl,yh, true);
  }
  
  Refresh();
}

void
PIScDrawWdg::Ptr2Move(int x, int y)
{
  SelForeground(PI_White);
  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);
  xEndDrag = x;
  yEndDrag = y;
  SelForeground(PI_Black);
  DrawBox(xBegDrag, yBegDrag, xEndDrag-xBegDrag, yEndDrag-yBegDrag);
}

//================================================================
// PIScDrawer
//================================================================

PIScDrawer::PIScDrawer()
: mDrawWdg(NULL)
{}

PIScDrawer::~PIScDrawer()
{
  if (mDrawWdg)
    mDrawWdg->RemoveDrawer(this);
}

void
PIScDrawer::AttachTo(PIScDrawWdg* wdg)
{
  if (mDrawWdg)
    mDrawWdg->RemoveDrawer(this);
  mDrawWdg = wdg;
}



//================================================================
// PIScFuncDrawer
//================================================================

PIScFuncDrawer::PIScFuncDrawer(PIScFuncDrawer::FUNC f)
: mFunc(f)
{}

PIScFuncDrawer::~PIScFuncDrawer()
{
}

void
PIScFuncDrawer::Draw()
{
  if (!mDrawWdg) return;
  double xMax = mDrawWdg->XMax();
  double xMin = mDrawWdg->XMin();
  double yMax = mDrawWdg->YMax();
  double yMin = mDrawWdg->YMin();
  double xStep = (xMax - xMin)/100;
  double xOld = xMin;
  double yOld = mFunc(xMin);
  for (double x = xMin+xStep; x<xMax; x+=xStep) {
    double y = mFunc(x);
    if (y>yMin && yOld>yMin && 
        y<yMax && yOld<yMax) 
           mDrawWdg->ScDrawLine(xOld, yOld, x, y);
    xOld = x;
    yOld = y;
  }
}

//================================================================
// PIScGFFDrawer
//================================================================

PIScGFFDrawer::PIScGFFDrawer(GeneralFitFunction* f)
: mFunc(f), mNParms(f->NPar()), mParms(new double[mNParms])
{}

PIScGFFDrawer::~PIScGFFDrawer()
{
  delete[] mParms;
}

void
PIScGFFDrawer::SetParms(double const*)
{}


void
PIScGFFDrawer::Draw()
{
}

