// This may look like C code, but it is really -*- C++ -*-

#ifndef PPERSIST_H_SEEN
#define PPERSIST_H_SEEN

// Flat file persistance, similar to Java serialization
//
// E. Aubourg     CEA DAPNIA/SPP  1999
// R. Ansari      LAL IN2P3/CNRS  03/2000


#include "machdefs.h"
#include "pexceptions.h"
#include "gnumd5.h"


#include <time.h>

#include <string>
#include <map>
#include <vector>
#include <typeinfo>


// Classe de base pour les objets qui peuvent devenir PPersist

namespace SOPHYA {
  
  class AnyDataObj;

  class PIOPersist;
  class PInPersist;
  class POutPersist;
  class PPersist;

  /* Persistant (delegate or mixin) object */

  class PPersist  {
  public:
    virtual           ~PPersist() {}
// J'ajoute cette fonction pour assurer la compatibilite 
// avec l'ancien PPersist d'Eros     (Reza 23/04/99)
    virtual int_4      ClassId() const { return(0); }

    void               Write(string const& fn) const;
    void               Read(string const& fn);

    virtual void       Write(POutPersist&) const;
    void               Read(PInPersist& s);               // Reads the type tag and the object
    void               Write(POutPersist&, string const& tag) const; 
    void               ReadAtTag(PInPersist& s, string const& tag);

    virtual AnyDataObj* DataObj()=0;       // Retourne l'objet reel 
    virtual void       SetDataObj(AnyDataObj &)=0;     

    virtual uint_8     getMemOId() const ;      // Renvoie l'identificateur de l'objet  - par defaut=0
    // Ces deux methodes doivent etre redefinies si   getMemOId() renvoie non nul (>0)
    virtual void       ShareDataReference(PPersist &);       
    virtual PPersist*  CloneSharedReference();
                                                // doit etre surcharge pour renvoyer un mem-oid correct
  protected:  
    virtual void       ReadSelf(PInPersist&)=0;           
    virtual void       WriteSelf(POutPersist&) const=0;

    friend class       PInPersist;
    friend class       POutPersist;
  };



  // Ancestor for PInPersist and POutPersist
  // Handles (statically) the registration of classes.

  class PIOPersist {
  public:
    enum                         {PPS_NATIVE = -1, PPS_LITTLE_ENDIAN = 0, PPS_BIG_ENDIAN = 1};
    typedef PPersist*            (*ClassCreatorFunc)();
  
    static void                  RegisterPPHandlerClass(uint_8 classId, string ppclass_name, ClassCreatorFunc f);
    static void                  RegisterDataObjClass(uint_8 classId, string class_name);

    static ClassCreatorFunc      FindCreatorFunc(uint_8 classId);

    static string                getPPClassName(uint_8 classId);
    static uint_8                getPPClassId(string const & typ_name);
    static uint_8                getPPClassId(PPersist const & ppo);
    static string                getDataObjClassName(uint_8 classId);
    static uint_8                getDataObjClassId(string const & typ_name);
    static uint_8                getDataObjClassId(AnyDataObj const & o);

    static uint_8 Hash(string const& typname) {
	md5_init(&ctx);
	md5_write(&ctx, (unsigned char*) typname.c_str(), typname.size());
	md5_final(&ctx);
	return ( *((uint_8*) ctx.buf) + *((uint_8*) (ctx.buf+8)));
    }
    static MD5_CONTEXT ctx;
  
    static void Initialize();   // Pour initialiser classList

    string FileName() { return filename; }   // Retourne le nom de fichier

  private:

    typedef map<uint_8, ClassCreatorFunc, less<uint_8> > ClassList;  
    // Pas de createur appele pour objets statiques sur Linux - $CHECK$ Reza 26/04/99
    static ClassList *           ppclassList;      // PPersist class list  
    static map<string, uint_8> * ppclassNameList;  // PPersist classId = f(PPersistClassName)
    static map<string, uint_8> * dobjclassNameList;  // PPersist classId = f(DataObjClassName)

  protected:

    enum {PPS_NULL = 0,             // this is a null object
	  PPS_STRING = 1,           // string, length (4b) + data
	  PPS_OBJECT = 2,           // classId, data...
	  PPS_REFERENCE = 3,        // objectId
          PPS_TAG = 4,              // tag entries
	  PPS_EOF = 5,              // Just before tag infomation, offset to PPS_TAG
	  PPS_ENDOBJECT = 6,        // marks the end of a given object information
          PPS_TAG_MARK = 7,         // To have a name tag, position marker in a file
          PPS_SIMPLE = 16,          // 16 + number of bytes, up to 8 bytes
          PPS_SIMPLE_ARRAY4 = 32,   // 32 + number of bytes, up to 8 bytes, then 4 bytes of length
          PPS_SIMPLE_ARRAY8 = 48    // 48 + number of bytes, up to 8 bytes, then 8 bytes of length
    };
  // The following values are used with PPS_SIMPLE and PPS_SIMPLE_ARRAY (Using OR)
   enum  {PPS_DATATYPE_CHAR = 0,        // 0  : DataType=character
          PPS_DATATYPE_FLOAT = 64,      // 64 : DataType=float 
          PPS_DATATYPE_INTEGER = 128,   // 128 :DataType=integer 
          PPS_DATATYPE_UNSIGNED = 192   // 192 :DataType=Unsigned integer 
   };

    map<string, int_8> tags;
    string filename;
  };


  // TBD : use hash tables instead of maps. Check hashtbl status in STL.

  class PInPersist : public PIOPersist {
  public:
    PInPersist(string const& flnm, bool scan=true);
    ~PInPersist();

    bool   GotoTag(string const& name);
    int    NbTags();
    bool   GotoTagNum(int itag);  // 0..NbTags-1
    string GetTagName(int itag);  // 0..NbTags-1
    string GetTagClassName(int itag);  // 0..NbTags-1
    vector<string> const &  GetTagNames();  

    void   GetByte (char& c);
    void   GetBytes(void* ptr, size_t bytes);
    void   GetR4   (r_4&);
    void   GetR4s  (r_4*, size_t);
    void   GetR8   (r_8&);
    void   GetR8s  (r_8*, size_t);
    void   GetI2   (int_2&);
    void   GetI2s  (int_2*, size_t);
    void   GetU2   (uint_2&);
    void   GetU2s  (uint_2*, size_t);
    void   GetI4   (int_4&);
    void   GetI4s  (int_4*, size_t);
    void   GetU4   (uint_4&);
    void   GetU4s  (uint_4*, size_t);
    void   GetI8   (int_8&);
    void   GetI8s  (int_8*, size_t);
    void   GetU8   (uint_8&);
    void   GetU8s  (uint_8*, size_t);
    void   GetLine (char* ptr, size_t len);
    void   GetStr  (string&);

    void   Get(char&   c) {GetByte(c);}
    void   Get(r_4&    x) {GetR4(x);}
    void   Get(r_8&    x) {GetR8(x);}
    void   Get(uint_2& x) {GetU2(x);}
    void   Get(int_2&  x) {GetI2(x);}
    void   Get(uint_4& x) {GetU4(x);}
    void   Get(int_4&  x) {GetI4(x);}
    void   Get(uint_8& x) {GetU8(x);}
    void   Get(int_8&  x) {GetI8(x);}
    void   Get(r_4*    x, size_t n) {GetR4s(x,n);}
    void   Get(r_8*    x, size_t n) {GetR8s(x,n);}
    void   Get(uint_2* x, size_t n) {GetU2s(x,n);}
    void   Get(int_2*  x, size_t n) {GetI2s(x,n);}
    void   Get(uint_4* x, size_t n) {GetU4s(x,n);}
    void   Get(int_4*  x, size_t n) {GetI4s(x,n);}
    void   Get(uint_8* x, size_t n) {GetU8s(x,n);}
    void   Get(int_8*  x, size_t n) {GetI8s(x,n);}
    void   Get(string& x) {GetStr(x);}
  
    //   Object Reading 
    PPersist*  ReadObject();
    void       GetObject(AnyDataObj & o);
    void       GetObject(AnyDataObj & o, string  tagname);
    PPersist*  GetPPObject(AnyDataObj * po=NULL);

    int    Version() {return version;}
    time_t CreationDate() { return creationdate; }
    string CreationDateStr(); 

    void   AnalyseTags(int lev=0);   // List (all or some) tags ...

    //  Reza 03/2000
    //  Methodes qui pourraient etre protected, mais doivent etre utilisables par PPersist 
    void   ReadReference(PPersist & ppo);  // Fill the ppo object from the reference tag
    PPersist * ReadReference();            // Creates object from the reference tag
    void   KeepOId(uint_8 oid, PPersist & ppo);        // Keeps the ppo in the objList

  protected:
    void   CheckTag   (short datasz, short datatype);
    void   CheckArrayTag(short datasz, size_t sz, short datatype);
    void   GetTypeTag (unsigned char& c);
    void   GetRawByte (char& c);
    void   GetRawUByte (unsigned char& c);
    void   GetRawBytes(void* ptr, size_t bytes);
    void   GetRawI2   (int_2&);
    void   GetRawI4   (int_4&);
    void   GetRawI8   (int_8&);
    void   GetRawU8   (uint_8&);

    void   Scan();

    istream* s;

    bool bigEndian;
    int version;

    time_t creationdate;

    // already read objects 
    typedef map<uint_8, PPersist * > ObjList;
    ObjList objList;
    // Si on a fait une lecture non sequentielle  -> seqread = false
    bool seqread;
    friend class PPersist;
  };

  class POutPersist : public PIOPersist {
  public:
    POutPersist(string const& flnm, int endianness = PPS_NATIVE);
    ~POutPersist();

    void WriteTag(string const& name);
  
    void PutByte (char c);
    void PutBytes(void const* ptr, size_t bytes);
    void PutR4   (r_4);
    void PutR4s  (r_4 const*, size_t);
    void PutR8   (r_8);
    void PutR8s  (r_8 const*, size_t);
    void PutI2   (int_2);
    void PutI2s  (int_2 const*, size_t);
    void PutU2   (uint_2);
    void PutU2s  (uint_2 const*, size_t);
    void PutI4   (int_4);
    void PutI4s  (int_4 const*, size_t);
    void PutU4   (uint_4);
    void PutU4s  (uint_4 const*, size_t);
    void PutI8   (int_8);
    void PutI8s  (int_8 const*, size_t);
    void PutU8   (uint_8);
    void PutU8s  (uint_8 const*, size_t);
    void PutLine (char const* ptr, size_t len=0); // deprecated ?
    void PutStr  (string const&);
    void PutPPObject (PPersist const*); // Like doing Write(stream) on PPersist object

    void PutObject(AnyDataObj & o);   // Creates the corresponding PPersist Object and call Write()
    void PutObject(AnyDataObj & o, string tagname);

    void   Put(char   c) {PutByte(c);}
    void   Put(r_4    x) {PutR4(x);}
    void   Put(r_8    x) {PutR8(x);}
    void   Put(uint_2 x) {PutU2(x);}
    void   Put(int_2  x) {PutI2(x);}
    void   Put(uint_4 x) {PutU4(x);}
    void   Put(int_4  x) {PutI4(x);}
    void   Put(uint_8 x) {PutU8(x);}
    void   Put(int_8  x) {PutI8(x);}
    void   Put(r_4 const*    x, size_t n) {PutR4s(x,n);}
    void   Put(r_8 const*    x, size_t n) {PutR8s(x,n);}
    void   Put(uint_2 const* x, size_t n) {PutU2s(x,n);}
    void   Put(int_2 const*  x, size_t n) {PutI2s(x,n);}
    void   Put(uint_4 const* x, size_t n) {PutU4s(x,n);}
    void   Put(int_4 const*  x, size_t n) {PutI4s(x,n);}
    void   Put(uint_8 const* x, size_t n) {PutU8s(x,n);}
    void   Put(int_8 const*  x, size_t n) {PutI8s(x,n);}
    void   Put(string const& s) {PutStr(s);}
    void   Put(PPersist const* x) {PutPPObject(x);}


  protected:
    ostream* s;
    bool bigEndian;

    void     PutArrayTag(short datasz, size_t sz, short datatype);
    void     PutRawByte (char);
    void     PutRawUByte (unsigned char);
    void     PutRawI2   (int_2);
    void     PutRawI4   (int_4);
    void     PutRawI8   (int_8);
    void     PutRawU8   (uint_8);
    void     PutRawBytes(void const* ptr, size_t bytes);
    bool     serializeNullAndRepeat(PPersist const* x);
    uint_8   findObjectId(PPersist const* x, int_8 & pos);
    uint_8   assignObjectId(PPersist const* x);

    // objreftag contains the assigned POutStream Object Id and the stream position 
    // of the original written object
    typedef struct { uint_8 ppsoid; int_8 ppspos; } objreftag;
    // already serialized objects are kept in a map as a function of the Objects memory Id
    typedef map<uint_8, objreftag, less<uint_8> >  ObjList;
    ObjList objList;
    uint_8 pps_OId;    // PPS Object Id 
  };
  
  
#define RAWPERSISTIO(_Type_,_xtyp_)                                             \
  inline POutPersist& operator << (POutPersist& c, _Type_ const& data)            \
  {                                                                               \
    c.Put##_xtyp_(data);                                                          \
    return c;                                                                     \
  }                                                                               \
                                                                                \
  inline PInPersist& operator >> (PInPersist& c, _Type_& data)                    \
  {                                                                               \
    c.Get##_xtyp_(data);                                                          \
    return c;                                                                     \
  }                                                                               

  RAWPERSISTIO(int_4,I4);
  RAWPERSISTIO(uint_4,U4);
  RAWPERSISTIO(int_2,I2);
  RAWPERSISTIO(uint_2,U2);
  RAWPERSISTIO(char,Byte);
  RAWPERSISTIO(r_4,R4);
  RAWPERSISTIO(r_8,R8);
  
#if 0
#define STRUCTPERSISTIO(_Type_, _field_, _size_)                                \
  inline POutPersist& operator << (POutPersist& c, _Type_ const& data)            \
  {                                                                               \
    c.PutBytes(&data._field_, _size_);                                            \
    return c;                                                                     \
  }                                                                               \
                                                                                \
  inline PInPersist& operator >> (PInPersist& c, _Type_& data)                    \
  {                                                                               \
    c.GetBytes(&data._field_, _size_);                                            \
    return c;                                                                     \
  }                                                                               

#endif
  

// --- Cela risque d'etre dangereux --- On le laisse au niveau des DataObj
//                           Reza 24/3/2000 
//   inline POutPersist& operator << (POutPersist& c, PPersist const& obj)
//     {
//       obj.Write(c);
//       return c;
//     }
  
//   inline PInPersist& operator >> (PInPersist& c, PPersist& obj)
//     {
//       obj.Read(c);
//       return c;
//     }

  // Utility class to
  //   - compute the class ID from a MD5 hash of the class name
  //   - register classes with PIOPersist, through PPRegister macro
  
  template <class T>
    class PPersistRegistrar {
    public:
      static PPersist* Create() {return new T();}
      static void Register(string id) { PIOPersist::RegisterPPHandlerClass(Hash(id), typeid(T).name(), Create); }
      static uint_8 Hash(string id) {
	return PIOPersist::Hash(id);
      }
    };
  
#define PPRegister(className) PPersistRegistrar<className>::Register(#className);
#define DObjRegister(ppclassName, className) PIOPersist::RegisterDataObjClass(PIOPersist::Hash(#ppclassName), typeid(className).name());

} // namespace

#endif
