//      template array class for numerical types
//                     R. Ansari, C.Magneville   03/2000

#include "machdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include "pexceptions.h"
#include "tarray.h"




// -------------------------------------------------------
//                Methodes de la classe
// -------------------------------------------------------

// Les constructeurs 
template <class T>
TArray<T>::TArray()
  : BaseArray() , mNDBlock() 
// Default constructor
{
}

template <class T>
TArray<T>::TArray(uint_4 ndim, const uint_4 * siz, uint_4 step)
  : BaseArray() , mNDBlock(ComputeTotalSize(ndim, siz, step, 1))
{
  string exmsg = "TArray<T>::TArray(uint_4, uint_4 *, uint_4)";
  if (!UpdateSizes(ndim, siz, step, 0, exmsg))  throw( ParmError(exmsg) );
}

template <class T>
TArray<T>::TArray(uint_4 nx, uint_4 ny, uint_4 nz, uint_4 nt, uint_4 nu)
  : BaseArray() , mNDBlock(nx*((ny>0)?ny:1)*((nz>0)?nz:1)*((nt>0)?nt:1)*((nu>0)?nu:1)) 
{
  uint_4 size[BASEARRAY_MAXNDIMS];
  size[0] = nx;   size[1] = ny;   size[2] = nz;
  size[3] = nt;   size[4] = nu;
  int ndim = 1;
  if ((size[1] > 0) && (size[2] > 0) && (size[3] > 0) && (size[4] > 0) ) ndim = 5;
  else if ((size[1] > 0) && (size[2] > 0) && (size[3] > 0) ) ndim = 4;
  else if ((size[1] > 0) && (size[2] > 0)) ndim = 3;
  else if (size[1] > 0)  ndim = 2;
  else ndim = 1;
  string exmsg = "TArray<T>::TArray(uint_4, uint_4, uint_4)";
  if (!UpdateSizes(ndim, size, 1, 0, exmsg))  throw( ParmError(exmsg) );
}

template <class T>
TArray<T>::TArray(uint_4 ndim, const uint_4 * siz, NDataBlock<T> & db, bool share, uint_4 step, uint_8 offset)
  : BaseArray() , mNDBlock(db, share) 
{
  string exmsg = "TArray<T>::TArray(uint_4, uint_4 *,  NDataBlock<T> & ... )";
  if (!UpdateSizes(ndim, siz, step, offset, exmsg))  throw( ParmError(exmsg) );
  
}

template <class T>
TArray<T>::TArray(uint_4 ndim, const uint_4 * siz, T* values, uint_4 step, uint_8 offset, Bridge* br)
  : BaseArray() , mNDBlock(ComputeTotalSize(ndim, siz, step, 1), values, br) 
{
  string exmsg = "TArray<T>::TArray(uint_4, uint_4 *, T* ... )";
  if (!UpdateSizes(ndim, siz, step, offset, exmsg))  throw( ParmError(exmsg) );
}

template <class T>
TArray<T>::TArray(const TArray<T>& a)
  : BaseArray() , mNDBlock(a.mNDBlock) 
{
  string exmsg = "TArray<T>::TArray(const TArray<T>&)";
  if (!UpdateSizes(a, exmsg))  throw( ParmError(exmsg) );
  if (a.mInfo) mInfo = new DVList(*(a.mInfo));
}

template <class T>
TArray<T>::TArray(const TArray<T>& a, bool share)
  : BaseArray() , mNDBlock(a.mNDBlock, share) 
{
  string exmsg = "TArray<T>::TArray(const TArray<T>&, bool)";
  if (!UpdateSizes(a, exmsg))  throw( ParmError(exmsg) );
  if (a.mInfo) mInfo = new DVList(*(a.mInfo));
}

// Destructeur 
template <class T>
TArray<T>::~TArray()
{
}

template <class T>
TArray<T>& TArray<T>::Set(const TArray<T>& a)
{
  if (this != &a) { 
    CloneOrShare(a);
    if (mInfo) delete mInfo;
    if (a.mInfo) mInfo = new DVList(*(a.mInfo));
  }
  return(*this);
}

template <class T>
void TArray<T>::Clone(const TArray<T>& a)
{
  string exmsg = "TArray<T>::Clone()";
  if (!UpdateSizes(a, exmsg))  throw( ParmError(exmsg) );
  mNDBlock.Clone(a.mNDBlock);
  if (mInfo) delete mInfo;
  if (a.mInfo) mInfo = new DVList(*(a.mInfo));
}

template <class T>
void TArray<T>::ReSize(uint_4 ndim, uint_4 * siz, uint_4 step)
{
  string exmsg = "TArray<T>::ReSize()";
  if (!UpdateSizes(ndim, siz, step, 0, exmsg))  throw( ParmError(exmsg) );
  mNDBlock.ReSize(totsize_);    
}

template <class T>
void TArray<T>::Realloc(uint_4 ndim, uint_4 * siz, uint_4 step, bool force)
{
  string exmsg = "TArray<T>::Realloc()";
  if (!UpdateSizes(ndim, siz, step, 0, exmsg))  throw( ParmError(exmsg) );
  mNDBlock.ReSize(totsize_);    
}


template <class T>
TArray<T>& TArray<T>::CompactAllDimensions()
{
  CompactAllDim();
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::CompactTrailingDimensions()
{
  CompactTrailingDim();
  return(*this);
}

template <class T>
double TArray<T>::ValueAtPosition(uint_8 ip) const
{
#ifdef SO_BOUNDCHECKING
  if (ip >= totsize_)  throw( ParmError("TArray<T>::ValueAtPosition(uint_8 ip) Out-of-bound Error") );
#endif
  return( (double)(*(mNDBlock.Begin()+Offset(ip))) );
}

// For complex values, we return the module of the complex number
double TArray< complex<r_4> >::ValueAtPosition(uint_8 ip) const
{
#ifdef SO_BOUNDCHECKING
  if (ip >= totsize_)  throw( ParmError("TArray<T>::ValueAtPosition(uint_8 ip) Out-of-bound Error") );
#endif
  complex<r_4> c = *(mNDBlock.Begin()+Offset(ip));
  double cr = (double)(c.real());
  double ci = (double)(c.imag());
  return( sqrt(cr*cr+ci*ci) );
}

double TArray< complex<r_8> >::ValueAtPosition(uint_8 ip) const
{
#ifdef SO_BOUNDCHECKING
  if (ip >= totsize_)  throw( ParmError("TArray<T>::ValueAtPosition(uint_8 ip) Out-of-bound Error") );
#endif
  complex<r_8> c = *(mNDBlock.Begin()+Offset(ip));
  double cr = (double)(c.real());
  double ci = (double)(c.imag());
  return( sqrt(cr*cr+ci*ci) );
}


template <class T>
TArray<T> TArray<T>::PackElements(bool force) const
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::PackElements() - Not Allocated Array ! ");
  if ( !force && (AvgStep() == 1) ) {
    TArray<T> ra(*this, true);
    ra.SetTemp(true);
    return(ra);
  }
  else {
    TArray<T> ra(ndim_, size_, 1);
    ra.CopyElt(*this);
    ra.SetTemp(true);
    return(ra);
  }
}

// SubArrays 
// $CHECK$ Reza 03/2000  Doit-on declarer cette methode const ?
template <class T>
TArray<T> TArray<T>::SubArray(Range rx, Range ry, Range rz, Range rt, Range ru) const
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::operator () (Range, ...) - Not Allocated Array ! ");
  uint_4 ndim = 0;
  uint_4 size[BASEARRAY_MAXNDIMS];
  uint_4 step[BASEARRAY_MAXNDIMS];
  uint_4 pos[BASEARRAY_MAXNDIMS];
  size[0] = rx.Size();
  size[1] = ry.Size();
  size[2] = rz.Size();
  size[3] = rt.Size();
  size[4] = ru.Size();

  step[0] = rx.Step();
  step[1] = ry.Step();
  step[2] = rz.Step();
  step[3] = rt.Step();
  step[4] = ru.Step();

  pos[0] = rx.Start();
  pos[1] = ry.Start();
  pos[2] = rz.Start();
  pos[3] = rt.Start();
  pos[4] = ru.Start();

  ndim = ndim_;
  TArray<T> ra;
  UpdateSubArraySizes(ra, ndim, size, pos, step); 
  ra.DataBlock().Share(this->DataBlock());
  ra.SetTemp(true);
  return(ra);
}

//   ...... Operation de calcul sur les tableaux ......
//              ------- Attention --------
//  Boucles normales prenant en compte les steps ....
//  Possibilite de // , vectorisation 
template <class T>
TArray<T>& TArray<T>::SetSeq(Sequence seq)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::SetSeq(Sequence ) - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    pe = Data();
    for(k=0; k<totsize_; k++ )  pe[k*step] = seq(k);
  }
  else {    // Non regular data spacing ...
    //    uint_4 ka = MaxSizeKA();
    uint_4 ka = 0;
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka);
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k++)  pe[k*step] = seq(j*gpas+k);
    }
  }
  return(*this);
}

//  >>>> Operations avec 2nd membre de type scalaire 

template <class T>
TArray<T>& TArray<T>::SetT(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::SetT(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] = x;
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] = x;
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::Add(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Add(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] += x;
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] += x;
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::Sub(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Sub(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] -= x;
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] -= x;
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::Mul(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Mul(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] *= x;
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] *= x;
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::Div(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Div(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] /= x;
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] /= x;
    }
  }
  return(*this);
}


template <class T>
TArray<T>& TArray<T>::SubInv(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::SubInv(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] = x-pe[k];
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] = x-pe[k];
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::DivInv(T x)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::DivInv(T )  - Not Allocated Array ! ");
  T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  pe[k] = x/pe[k];
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  pe[k] = x/pe[k];
    }
  }
  return(*this);
}


//  >>>> Operations avec 2nd membre de type tableau
template <class T>
TArray<T>& TArray<T>::AddElt(const TArray<T>& a)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::AddElt(const TArray<T>& )  - Not Allocated Array ! ");
  if (!CompareSizes(a)) 
    throw(SzMismatchError("TArray<T>::AddElt(const TArray<T>&) SizeMismatch")) ;

  T * pe;
  const T * pea;
  uint_8 j,k,ka;
  if ((AvgStep() > 0) && (a.AvgStep() > 0) )   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 stepa = a.AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    pea = a.Data();
    for(k=0, ka=0;  k<maxx;  k+=step, ka+=stepa )  pe[k] += pea[ka] ;
  }
  else {    // Non regular data spacing ...
    uint_4 ax = MaxSizeKA();
    uint_8 step = Step(ax);
    uint_8 stepa = a.Step(ax);
    uint_8 gpas = Size(ax)*step;
    uint_8 naxa = Size()/Size(ax);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ax,j);
      pea = a.DataBlock().Begin()+a.Offset(ax,j);
      for(k=0, ka=0;  k<gpas;  k+=step, ka+=stepa)  pe[k] += pea[ka]; 
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::SubElt(const TArray<T>& a)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::SubElt(const TArray<T>& )  - Not Allocated Array ! ");
  if (!CompareSizes(a)) 
    throw(SzMismatchError("TArray<T>::SubElt(const TArray<T>&) SizeMismatch")) ;

  T * pe;
  const T * pea;
  uint_8 j,k,ka;
  if ((AvgStep() > 0) && (a.AvgStep() > 0) )   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 stepa = a.AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    pea = a.Data();
    for(k=0, ka=0;  k<maxx;  k+=step, ka+=stepa )  pe[k] -= pea[ka] ;
  }
  else {    // Non regular data spacing ...
    uint_4 ax = MaxSizeKA();
    uint_8 step = Step(ax);
    uint_8 stepa = a.Step(ax);
    uint_8 gpas = Size(ax)*step;
    uint_8 naxa = Size()/Size(ax);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ax,j);
      pea = a.DataBlock().Begin()+a.Offset(ax,j);
      for(k=0, ka=0;  k<gpas;  k+=step, ka+=stepa)  pe[k] -= pea[ka]; 
    }
  }
  return(*this);
}


template <class T>
TArray<T>& TArray<T>::MulElt(const TArray<T>& a)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::MulElt(const TArray<T>& )  - Not Allocated Array ! ");
  if (!CompareSizes(a)) 
    throw(SzMismatchError("TArray<T>::MulElt(const TArray<T>&) SizeMismatch")) ;

  T * pe;
  const T * pea;
  uint_8 j,k,ka;
  if ((AvgStep() > 0) && (a.AvgStep() > 0) )   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 stepa = a.AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    pea = a.Data();
    for(k=0, ka=0;  k<maxx;  k+=step, ka+=stepa )  pe[k] *= pea[ka] ;
  }
  else {    // Non regular data spacing ...
    uint_4 ax = MaxSizeKA();
    uint_8 step = Step(ax);
    uint_8 stepa = a.Step(ax);
    uint_8 gpas = Size(ax)*step;
    uint_8 naxa = Size()/Size(ax);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ax,j);
      pea = a.DataBlock().Begin()+a.Offset(ax,j);
      for(k=0, ka=0;  k<gpas;  k+=step, ka+=stepa)  pe[k] *= pea[ka]; 
    }
  }
  return(*this);
}


template <class T>
TArray<T>& TArray<T>::DivElt(const TArray<T>& a)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::DivElt(const TArray<T>& )  - Not Allocated Array ! ");
  if (!CompareSizes(a)) 
    throw(SzMismatchError("TArray<T>::DivElt(const TArray<T>&) SizeMismatch")) ;

  T * pe;
  const T * pea;
  uint_8 j,k,ka;
  if ((AvgStep() > 0) && (a.AvgStep() > 0) )   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 stepa = a.AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    pea = a.Data();
    for(k=0, ka=0;  k<maxx;  k+=step, ka+=stepa )  pe[k] /= pea[ka] ;
  }
  else {    // Non regular data spacing ...
    uint_4 ax = MaxSizeKA();
    uint_8 step = Step(ax);
    uint_8 stepa = a.Step(ax);
    uint_8 gpas = Size(ax)*step;
    uint_8 naxa = Size()/Size(ax);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ax,j);
      pea = a.DataBlock().Begin()+a.Offset(ax,j);
      for(k=0, ka=0;  k<gpas;  k+=step, ka+=stepa)  pe[k] /= pea[ka]; 
    }
  }
  return(*this);
}

template <class T>
TArray<T>& TArray<T>::CopyElt(const TArray<T>& a)
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::CopyElt(const TArray<T>& )  - Not Allocated Array ! ");
  if (!CompareSizes(a)) 
    throw(SzMismatchError("TArray<T>::MultElt(const TArray<T>&) SizeMismatch")) ;

  T * pe;
  const T * pea;
  uint_8 j,k,ka;
  if ((AvgStep() > 0) && (a.AvgStep() > 0) )   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 stepa = a.AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    pea = a.Data();
    for(k=0, ka=0;  k<maxx;  k+=step, ka+=stepa )  pe[k] = pea[ka] ;
  }
  else {    // Non regular data spacing ...
    uint_4 ax = MaxSizeKA();
    uint_8 step = Step(ax);
    uint_8 stepa = a.Step(ax);
    uint_8 gpas = Size(ax)*step;
    uint_8 naxa = Size()/Size(ax);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ax,j);
      pea = a.DataBlock().Begin()+a.Offset(ax,j);
      for(k=0, ka=0;  k<gpas;  k+=step, ka+=stepa)  pe[k] = pea[ka]; 
    }
  }
  return(*this);
}


// Somme et produit des elements
template <class T>
T TArray<T>::Sum() const
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Sum()  - Not Allocated Array ! ");
  T ret=0;
  const T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  ret += pe[k];
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  ret += pe[k] ;
    }
  }
  return ret;
}

template <class T>
T TArray<T>::Product() const
{
  if (NbDimensions() < 1) 
    throw RangeCheckError("TArray<T>::Product()  - Not Allocated Array ! ");
  T ret=0;
  const T * pe;
  uint_8 j,k;
  if (AvgStep() > 0)   {  // regularly spaced elements
    uint_8 step = AvgStep();
    uint_8 maxx = totsize_*step;
    pe = Data();
    for(k=0; k<maxx; k+=step )  ret *= pe[k];
  }
  else {    // Non regular data spacing ...
    uint_4 ka = MaxSizeKA();
    uint_8 step = Step(ka);
    uint_8 gpas = Size(ka)*step;
    uint_8 naxa = Size()/Size(ka);
    for(j=0; j<naxa; j++)  {
      pe = mNDBlock.Begin()+Offset(ka,j);
      for(k=0; k<gpas; k+=step)  ret *= pe[k] ;
    }
  }
  return ret;
}



// ----------------------------------------------------
//       Impression, etc ...
// ----------------------------------------------------

template <class T>
string TArray<T>::InfoString() const
{
  string rs = "TArray<" ;
  rs += typeid(T).name();
  rs += "> ";
  return(rs);
}

template <class T>
void TArray<T>::Print(ostream& os, int_4 maxprt, bool si) const
{
  if (maxprt < 0)  maxprt = max_nprt_;
  int_4 npr = 0;
  Show(os, si);
  if (ndim_ < 1) return;
  uint_4 k0,k1,k2,k3,k4;
  for(k4=0; k4<size_[4]; k4++) {
    if (size_[4] > 1) cout << "\n ----- Dimension 5 (U) K4= " << k4 << endl;
    for(k3=0; k3<size_[3]; k3++) {
      if (size_[3] > 1) cout << "\n ----- Dimension 4 (T) K3= " << k3 << endl;
      for(k2=0; k2<size_[2]; k2++) {
	if (size_[2] > 1) cout << "\n ----- Dimension 3 (Z) K2= " << k2 << endl;
	for(k1=0; k1<size_[1]; k1++) {
	  if ( (size_[1] > 1) && (size_[0] > 10) ) cout << "----- Dimension 2 (Y) K1= " << k1 << endl;
	  for(k0=0; k0<size_[0]; k0++) {
            if(k0 > 0) os << ", ";  
            os << Elem(k0, k1, k2, k3, k4);     npr++;
	    if (npr >= maxprt) {
	      if (npr < totsize_)  os << "\n     .... " << endl; return;
	    }
	  }
          os << endl;
	}
      }
    }
  }
  os <<  endl;
}


template <class T>
void TArray<T>::CloneOrShare(const TArray<T>& a)
{
  string exmsg = "TArray<T>::CloneOrShare()";
  if (!UpdateSizes(a.ndim_, a.size_, a.step_, a.offset_, exmsg))  throw( ParmError(exmsg) );
  mNDBlock.CloneOrShare(a.mNDBlock);
}

template <class T>
void TArray<T>::Share(const TArray<T>& a)
{
  string exmsg = "TArray<T>::Share()";
  if (!UpdateSizes(a.ndim_, a.size_, a.step_, a.offset_, exmsg))  throw( ParmError(exmsg) );
  mNDBlock.Share(a.mNDBlock);
}



///////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////
#ifdef __CXX_PRAGMA_TEMPLATES__
/*
#pragma define_template TArray<uint_1>
#pragma define_template TArray<int_2>
#pragma define_template TArray<uint_4>
#pragma define_template TArray<uint_8>
*/
#pragma define_template TArray<uint_2>
#pragma define_template TArray<int_4>
#pragma define_template TArray<int_8>
#pragma define_template TArray<r_4>
#pragma define_template TArray<r_8>
#pragma define_template TArray< complex<r_4> > 
#pragma define_template TArray< complex<r_8> > 
#endif

#if defined(ANSI_TEMPLATES) || defined(GNU_TEMPLATES)
/*
template class TArray<uint_1>; 
template class TArray<int_2>;  
template class TArray<uint_4>;
template class TArray<uint_8>;
*/
template class TArray<uint_2>;
template class TArray<int_4>;
template class TArray<int_8>;
template class TArray<r_4>;
template class TArray<r_8>;
template class TArray< complex<r_4> >;
template class TArray< complex<r_8> >;
#endif


