#ifndef FITSFILE_H
#define FITSFILE_H

#include "ndatablock.h"
#include "dvlist.h"
#include "FitsIO/fitsio.h" 

#define OPENFILE    0
#define CREATEFILE  1
#define LEN_KEYWORD 9

namespace SOPHYA {

//
//! Virtual Class for managing Interface for SOPHYA objects to FITS Format Files (uses cfitsio lib)

/*! 

Each SOPHYA object XXX is associated with a object of class FITS_XXX 
 (inheriting from FitsFile), to which input/output operations with FITS 
 files are delegated. A typical example of use is the following : 

\verbatim
  int m=... ;
  SphereHEALPix<r_8> sphere1(m);           // definition of the SOPHYA object
  .... fill the sphere ....

  FITS_SphereHEALPix<r_8> fits_sph1(sphere1); 
                                           // delegated object
  fits_sph.Write("myfile.fits");           // writing on FITS file

   FITS_SphereHEALPix<r_8> fits_sph2("myfile.fits"); 
                                           // load a delegated object 
                                           // from FITS file
   SphereHEALPix<r_8> sphere2=(SphereHEALPix<r_8>)fits_sph2; 
                                           // casting the delegated object
                                           // into a SOPHYA object
\endverbatim

*/
class FitsFile
{

public:

  FitsFile();
  virtual ~FitsFile();
  static int NbBlocks(char flnm[]);
  static void getBlockType(char flnm[], int hdunum, string& typeOfExtension, int& naxis, vector<int>& naxisn, string& dataType, DVList& dvl  );
/*! return true if the current header  corresponds to a FITS image extension */
inline bool IsFitsImage() const { return (hdutype_ == IMAGE_HDU);}

/*! return true if the current header  corresponds to a FITS ASCII or BINTABLE extension */
inline bool IsFitsTable() const {return (hdutype_ == ASCII_TBL || hdutype_ == BINARY_TBL);}

/*!
this method is called from inherited objects : 

moves to header 'hdunum' on file 'flnm'

gets parameters in header 

calls the method 'ReadFromFits' from the inherited delegated object

*/ 
  void   ReadF(char flnm[],int hdunum= 0);
 
/*!
this method is called from inherited objects : 

opens a file 'flnm' 

gets parameters in header 

calls the method 'ReadFromFits' from the inherited delegated object

*/ 
  void    WriteF(char flnm[],int hdunum= 0); 


  /*!
fill the array 'map' with double data from the current extension on FITS file.
If the extension is BINTABLE, the first column is provided. 

\param <nentries>  number of data to be read
  */
  void    GetSingleColumn(double* map, int nentries) const;

  /*! same as above with float data */
  void    GetSingleColumn(float*  map, int nentries) const;

  /*! same as above with int data */
  void    GetSingleColumn(int* map, int nentries) const;

  /*! number of dimensions of an image extension : NAXIS parameter (in FITS notations)
   */
inline int nbDimOfImage() const {return naxis_;}

/*!
 total number of data in the current IMAGE extension
 */
inline int nbOfImageData() const { return nbData_; }

/*! a reference on a vector containing sizes of the NAXIS dimensions : NAXIS1, NAXIS2, NAXIS3 wtc.
 */
 inline const vector<int>& dimOfImageAxes() const { return naxisn_;}

 /*! return number of columns (return 1 if IMAGE) */
  int     NbColsFromFits() const;

  /*! return a character denoting data type of column number 'nocol' in a BINTABLE :

D : double

E : float 

I : integer
    
S : character string 

  */
  char    ColTypeFromFits(int nocol) const;

  /*! number of data in the current IMAGE extension on FITS file, or number
 of data of column number 'nocol' of the current BINTABLE extension
  */
  int     NentriesFromFits(int nocol) const;

  /*! name of the column number 'nocol' of the current BINTABLE extension
   */
  string  ColNameFromFits(int nocol) const;

  
  /*! number of characters of each data  for the column number 'nocol' (if char* typed) of the current BINTABLE extension
   */
  int     ColStringLengthFromFits(int nocol) const;

  /*! return a reference on a DVList containing the keywords from FITS file
   */
  inline const DVList& DVListFromFits() const { return dvl_;}


  /*!
fill the array 'valeurs' with double data from the current BINTABLE extension on FITS file, from column number 'NoCol'

\param <nentries>  number of data to be read
  */
  void GetBinTabFCol(double* valeurs, int nentries, int NoCol) const;

  /*! same as previous method with float data */
  void GetBinTabFCol(float* valeurs, int nentries, int NoCol) const;
  /*! same as previous method with int data */
  void GetBinTabFCol(int* valeurs, int nentries,  int NoCol) const;
  /*! same as previous method with char* data */
  void GetBinTabFCol(char** valeurs,int nentries, int NoCol) const;
  // Write elements into the FITS data array

  /*! write double data from array 'map'on an IMAGE extension 
\param <nbData>  number of data to be written

   */
  void putImageToFits( int nbData, double* map) const;

  /*! same as previous method with float data */
  void putImageToFits(int nbData, float* map ) const;

  /*! same as previous method with int data */
  void putImageToFits(int nbData, int* map) const;

  /*! write double data from array 'donnees ' on column number 'nocol' of a BINTABLE  extension.
\param <nentries>  number of data to be written

   */
  void putColToFits(int nocol, int nentries, double* donnees) const;

  /*! same as previous method with float data */
  void putColToFits(int nocol, int nentries, float* donnees) const;

  /*! same as previous method with int data */
  void putColToFits(int nocol, int nentries, int* donnees) const;

  /*! same as previous method with char* data */
  void putColToFits(int nocol, int nentries, char** donnees) const;

  /*! create an IMAGE header on FITS file.
\param <type> type of data (see method ColTypeFromFits)
\param <nbdim>  number of dimensions : 1D, 2D, 3D etc. = NAXIS
\param <naxisn>  array containind sizes of the different dimensions
  */
  void makeHeaderImageOnFits(char type, int nbdim, int* naxisn) const;

  /*! create an BINTABLE header on FITS file.
\param <fieldType> array containing characters denoting types of the different column (see method ColTypeFromFits)
\param <Noms>  array of the names of columns
\param <nentries>  number of data of each column
\param <tfields> number of columns
\param <dvl> a SOPHYA DVList containing keywords to be appended
\param <extname> keyword EXTNAME for FITS file
\param <taille_des_chaines> vector containing the number of characters of  data  for each char* typed column, with order of appearance in 'fieldType'
   */
  void makeHeaderBntblOnFits ( char* fieldType, char** Noms, int nentries, int tfields, DVList &dvl, char* extname,  vector<int> taille_des_chaines) const;

  void readheader();
static  void printerror(int&) ;
static  void printerror(int&,char* texte) ;
  protected:  
    virtual void    ReadFromFits(const FitsFile& ff)=0;           
    virtual void    WriteToFits(const FitsFile& ff)=0;           
private:

static  void GetImageParameters (fitsfile* fileptr,int& bitpix,int& naxis,vector<int>& naxisn);
static  void GetBinTabParameters(fitsfile* fileptr, int& nbcols, int& nrows,
				  vector<int>& repeat,
				  vector<string>& noms, 
				  vector<char>& types,   
				  vector<int>&  taille_des_chaines);
static void KeywordsIntoDVList(fitsfile* fileptr, DVList& dvl); 
  void writeSignatureOnFits() const; 

  //! pointer to the FITS file, defined in fitsio.h 
  fitsfile *fptr_; 
  
  //!  image or bintable ?
  int hdutype_;

//! index of header to be read/written
  int hdunum_;  


  //! fits-Image parameter 
  int bitpix_;

  //! fits-Image parameter 
  int naxis_;

  //! fits-Image parameters : sizes of dimensions
  vector<int> naxisn_;

  //! fits-Image parameter: number of data 
  int nbData_;

  //! Bintable parameter
  int nrows_;

  //! Bintable parameter
  vector<int> repeat_;

  //! Bintable parameter
  int nbcols_;

  //! Bintable parameter: column names
  vector<string> noms_; 
 
  //! Bintable parameters: types of columns (D: double, E: float, I: integers,  A: char*)
  vector<char> types_;   
    
  //! Bintable parameters:   length of the char* variables                 
  vector<int>  taille_des_chaines_;

  //! DVList for transferring keywords
  DVList dvl_;
};


} // Fin du namespace


#endif
