#ifndef HISPROF_SEEN
#define HISPROF_SEEN

#include <stdio.h>
#include "peida.h"
#include "tvector.h"
#include "ppersist.h"
#include "histos.h"

namespace SOPHYA {

/*!
  Classe de profil d'histogrammes.
*/
class HProf : public Histo {
  friend class ObjFileIO<HProf>;
public:

  // CREATOR / DESTRUCTOR
  HProf();
  HProf(float xMin, float xMax, int nBin=100, float yMin=1., float yMax=-1.);
  HProf(const HProf& H);
  virtual ~HProf();

  // UPDATING or SETTING
  void   Zero();
  void   UpdateHisto() const;
  void   SetErrOpt(bool spread = true);
  void   Add(float x, float y, float w = 1.);
  void   AddBin(int numBin, float y, float w = 1.);
  
  // Acces a l information
  //! Retourne l'histogramme de profil.
  inline Histo GetHisto() {if(!Ok) UpdateHisto(); return (Histo) *this;}
  //! Retourne le contenu de la moyenne dans le vecteur v
  inline void GetMean(Vector& v) {if(!Ok) UpdateHisto(); Histo::GetValue(v);}
  //! Retourne le contenu au carre de la dispersion/erreur dans le vecteur v
  inline void GetError2(Vector& v) {if(!Ok) UpdateHisto(); Histo::GetError2(v);}
  //! Retourne le contenu du bin i
  inline float operator()(int i) const {if(!Ok) UpdateHisto(); return data[i];}
  //! Retourne le carre de la dispersion/erreur du bin i
  inline float Error2(int i) const {if(!Ok) UpdateHisto(); return (float) err2[i];}
  //! Retourne la dispersion/erreur du bin i
  inline float Error(int i) const
         {if(!Ok) UpdateHisto(); return err2[i]>0. ? (float) sqrt(err2[i]) : 0.f;}

  // Operators
  HProf& operator = (const HProf& h);
  HProf& operator += (const HProf& a);
  friend HProf operator + (const HProf& a, const HProf& b);

  // Fit
  //! Fit du profile par ``gfit''.
  inline int    Fit(GeneralFit& gfit)
         {if(!Ok) UpdateHisto(); return Histo::Fit(gfit,0);}
  //! Retourne l'Histogramme des residus par ``gfit''.
  inline Histo  FitResidus(GeneralFit& gfit)
         {if(!Ok) UpdateHisto(); return Histo::FitResidus(gfit);}
  //! Retourne l'Histogramme de la fonction fittee par ``gfit''.
  inline Histo  FitFunction(GeneralFit& gfit)
         {if(!Ok) UpdateHisto(); return Histo::FitFunction(gfit);}

  // Print
  //! Print, voir detail dans Histo::Print
  inline void Print(int dyn=100,float hmin=1.,float hmax=-1.,int pflag=0,int il=1,int ih=-1)
                           {if(!Ok) UpdateHisto(); Histo::Print(dyn,hmin,hmax,pflag,il,ih);}

protected:
  void Delete();

  double*        SumY;  //!< somme
  double*        SumY2; //!< somme des carres
  double*        SumW;  //!< somme des poids
  bool           Ok;    //!< true isiupdate fait
  float          YMin;  //!< limite minimum Y pour somme
  float          YMax;  //!< limite maximum Y pour somme
  uint_2         Opt;   //!< options pour les erreurs
};


inline POutPersist& operator << (POutPersist& os, HProf & obj)
{ ObjFileIO<HProf> fio(&obj);  fio.Write(os);  return(os); }
inline PInPersist& operator >> (PInPersist& is, HProf & obj)
{ ObjFileIO<HProf> fio(&obj);  fio.Read(is);  return(is); }
// Classe pour la gestion de persistance
// ObjFileIO<HProf>


} // Fin du namespace

#endif // HISPROF_SEEN
