// This may look like C code, but it is really -*- C++ -*-
//      template array class for numerical types
//                     R. Ansari, C.Magneville   03/2000

#ifndef TArray_SEEN
#define TArray_SEEN

#include "machdefs.h"
#include <math.h>
#include <iostream.h>
#include "basarr.h"
#include "ndatablock.h"
#include <complex>
#include "utilarr.h"


namespace SOPHYA {

// Forward declaration
template <class T> class FIO_TArray;

//   --------------------------- classe template Array -----------------------
//  ( See BaseArray class for data organisation in  memory and related methods )

//! Class for template arrays
template <class T>
class TArray : public BaseArray {
public:
  // Creation / destruction 
  TArray();
  TArray(uint_4 ndim, const uint_4 * siz, uint_4 step =1);
  TArray(uint_4 nx, uint_4 ny=0, uint_4 nz=0, uint_4 nt=0, uint_4 nu=0);
  TArray(uint_4 ndim, const uint_4 * siz, NDataBlock<T> & db, bool share=false, uint_4 step=1, uint_8 offset=0);
  TArray(uint_4 ndim, const uint_4 * siz, T* values, uint_4 step=1, uint_8 offset=0, Bridge* br=NULL);
  TArray(const TArray<T>& a);
  TArray(const TArray<T>& a, bool share);

  virtual ~TArray();

  // A = B : partage les donnees si "a" est temporaire, clone sinon.
  //! = operator between TArray
  /*! \sa Set */
  inline  TArray<T>& operator = (const TArray<T>& a) { return Set(a); }
  virtual TArray<T>& Set(const TArray<T>& a);

  // Gestion taille/Remplissage
  virtual void Clone(const TArray<T>& a); 
  void ReSize(uint_4 ndim, uint_4 * siz, uint_4 step=1);
  void Realloc(uint_4 ndim, uint_4 * siz, uint_4 step=1, bool force=false);

  // Compacts size=1 array dimensions
  virtual TArray<T>& CompactAllDimensions();
  virtual TArray<T>& CompactTrailingDimensions();

  // Packing array elements in memory
  virtual TArray<T> PackElements(bool force=false) const ;

  // SubArrays - $CHECK$ Reza 03/2000 je ne sais pas s'il faut declarer ca const ??
  TArray<T> SubArray(Range rx, Range ry, Range rz, Range rt, Range ru) const ;
  //! () operator for Sub arrays extraction
  /*! \sa SubArray */
  inline TArray<T> operator () (Range rx, Range ry, Range rz, Range rt=0, Range ru=0) const
                   { return  SubArray(rx, ry, rz, rt, ru); }

  // ---- Access to data
  // Definition of virtual element acces method inherited from BaseArray class 
  virtual double ValueAtPosition(uint_8 ip) const;

  // Data Access: operator overloaded inline acces methods 
  inline T const& operator()(uint_4 ix, uint_4 iy, uint_4 iz) const ;
  inline T&       operator()(uint_4 ix, uint_4 iy, uint_4 iz);
  inline T const& operator()(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu=0) const ;
  inline T&       operator()(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu=0);
  inline T const& operator[](uint_8 ip) const ;
  inline T&       operator[](uint_8 ip);

  inline T const& Elem(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it=0, uint_4 iu=0) const ;
  inline T&       Elem(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it=0, uint_4 iu=0);
  inline T const& ElemCheckBound(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it=0, uint_4 iu=0) const ;
  inline T&       ElemCheckBound(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it=0, uint_4 iu=0);

  //! Return pointer to first element adress
  inline        T* Data()       {return mNDBlock.Begin()+offset_;}
  //! Return pointer to first element adress
  inline const  T* Data() const {return mNDBlock.Begin()+offset_;}
  //! Return reference to datablock NDataBlock
  inline        NDataBlock<T>& DataBlock()       {return mNDBlock;}
  //! Return reference to datablock NDataBlock
  inline const  NDataBlock<T>& DataBlock() const {return mNDBlock;}

  // Temporaire?
  //! Are the array temporay ?
  inline bool   IsTemp(void) const {return mNDBlock.IsTemp();}
  //! Set the array as temporay
  inline void   SetTemp(bool temp=false) const {mNDBlock.SetTemp(temp);}

// Operations diverses  = , +=, ...
// Conversion en type T, if Size() == 1
  inline T toScalar();
// Met les elements a une suite de valeurs
  virtual TArray<T>&  SetSeq(Sequence seq);
  //! Fill TArray with Sequence \b seq
  inline  TArray<T>&  operator = (Sequence seq)    { return SetSeq(seq); }
// A = x (tous les elements a x)
  virtual TArray<T>&  SetT(T x);
  //! Fill TArray with all elements equal to \b x
  inline  TArray<T>&  operator = (T x)             { return SetT(x); }
// A += -= *= /= x (ajoute, soustrait, ... x a tous les elements)
  virtual TArray<T>&  Add(T x);
  //! Add \b x to all elements
  inline  TArray<T>&  operator += (T x)            { return Add(x); }
  virtual TArray<T>&  Sub(T x);
  //! Substract \b x to all elements
  inline  TArray<T>&  operator -= (T x)            { return Sub(x); }
  virtual TArray<T>&  Mul(T x);
  //! Multiply all elements by \b x
  inline  TArray<T>&  operator *= (T x)            { return Mul(x); }
  virtual TArray<T>&  Div(T x);
  //! Divide all elements by \b x
  inline  TArray<T>&  operator /= (T x)            { return Div(x); }
  virtual TArray<T>&  SubInv(T x);  //   A ---> x-A
  virtual TArray<T>&  DivInv(T x);  //   A ---> x/A

// A += -=  (ajoute, soustrait element par element les deux tableaux )
  virtual TArray<T>&  AddElt(const TArray<T>& a);
  //! Operator TArray += TArray
  inline  TArray<T>&  operator += (const TArray<T>& a)  { return AddElt(a); }
  virtual TArray<T>&  SubElt(const TArray<T>& a);
  //! Operator TArray -= TArray
  inline  TArray<T>&  operator -= (const TArray<T>& a)  { return SubElt(a); }
// Multiplication, division element par element les deux tableaux 
  virtual TArray<T>&  MulElt(const TArray<T>& a);
  virtual TArray<T>&  DivElt(const TArray<T>& a);
// Recopie des valeurs, element par element
  virtual TArray<T>&  CopyElt(const TArray<T>& a);

// Somme et produit des elements
  virtual T Sum() const ;
  virtual T Product() const ;

// Impression, I/O, ...
  virtual string InfoString() const;   
  virtual void   Print(ostream& os, int_4 maxprt=-1, bool si=false) const ;

//  Pour la gestion de persistance
  friend class  FIO_TArray<T>;

protected:
  // partage les donnees si "a" temporaire, clone sinon.
  void CloneOrShare(const TArray<T>& a);
  // Share: partage les donnees de "a"
  void Share(const TArray<T>& a);

  NDataBlock<T> mNDBlock; //!< Block for datas
};

////////////////////////////////////////////////////////////////
//   Surcharge d'operateur << 
//! Print TArray \b a on stream \b os
template <class T>
inline ostream& operator << (ostream& os, const TArray<T>& a)
                            { a.Print(os);    return(os);    }

////////////////////////////////////////////////////////////////
// Surcharge d'operateurs A (+,-,*,/) (T) x

/*! \ingroup TArray \fn operator+(const TArray<T>&,T)
  \brief Operator TArray = TArray + constant */
template <class T> inline TArray<T> operator + (const TArray<T>& a, T b)
    {TArray<T> result(a); result.SetTemp(true); result.Add(b); return result;}

/*! \ingroup TArray \fn operator+(T,const TArray<T>&)
  \brief Operator TArray = constant + TArray */
template <class T> inline TArray<T> operator + (T b,const TArray<T>& a)
    {TArray<T> result(a); result.SetTemp(true); result.Add(b); return result;}

/*! \ingroup TArray \fn operator-(const TArray<T>&,T)
  \brief Operator TArray = TArray - constant */
template <class T> inline TArray<T> operator - (const TArray<T>& a, T b)
    {TArray<T> result(a); result.SetTemp(true); result.Sub(b); return result;}

/*! \ingroup TArray \fn operator-(T,const TArray<T>&)
  \brief Operator TArray = constant - TArray */
template <class T> inline TArray<T> operator - (T b,const TArray<T>& a)
    {TArray<T> result(a); result.SetTemp(true); result.SubInv(b); return result;}

/*! \ingroup TArray \fn operator*(const TArray<T>&,T)
  \brief Operator TArray = TArray * constant */
template <class T> inline TArray<T> operator * (const TArray<T>& a, T b)
    {TArray<T> result(a); result.SetTemp(true); result.Mul(b); return result;}

/*! \ingroup TArray \fn operator*(T,const TArray<T>&)
  \brief Operator TArray = constant * TArray */
template <class T> inline TArray<T> operator * (T b,const TArray<T>& a)
    {TArray<T> result(a); result.SetTemp(true); result.Mul(b); return result;}

/*! \ingroup TArray \fn operator/(const TArray<T>&,T)
  \brief Operator TArray = TArray / constant */
template <class T> inline TArray<T> operator / (const TArray<T>& a, T b)
    {TArray<T> result(a); result.SetTemp(true); result.DivInv(b); return result;}

////////////////////////////////////////////////////////////////
// Surcharge d'operateurs C = A (+,-) B

/*! \ingroup TArray \fn operator+(const TArray<T>&,const TArray<T>&)
  \brief Operator TArray = TArray + TArray */
template <class T>
inline TArray<T> operator + (const TArray<T>& a,const TArray<T>& b)
    {TArray<T> result(a); result.SetTemp(true); result.AddElt(b); return result;}

/*! \ingroup TArray \fn operator-(const TArray<T>&,const TArray<T>&)
  \brief Operator TArray = TArray - TArray */
template <class T>
inline TArray<T> operator - (const TArray<T>& a,const TArray<T>& b)
    {TArray<T> result(a); result.SetTemp(true); result.SubElt(b); return result;}


// --------------------------------------------------
//        inline element acces methods
// --------------------------------------------------

//! Return element (ix,iy,iz,it,iu) value
template <class T>
inline T const& TArray<T>::Elem(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu) const
{
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2] +
	      it*step_[3] + iu*step_[4]) );
}

//! Return element (ix,iy,iz,it,iu) value
template <class T>
inline T & TArray<T>::Elem(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu)
{
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2] +
	      it*step_[3] + iu*step_[4]) );
}

//! Return element (ix,iy,iz,it,iu) value with Check of indexes bound first
template <class T>
inline T const& TArray<T>::ElemCheckBound(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu) const
{
  CheckBound(ix, iy, iz, it, iu, 4);
  return(Elem(ix, iy, iz, it, iu));
}

//! Return element (ix,iy,iz,it,iu) value with Check of indexes bound first
template <class T>
inline T & TArray<T>::ElemCheckBound(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu)
{
  CheckBound(ix, iy, iz, it, iu, 4);
  return(Elem(ix, iy, iz, it, iu));
}

//! Return element (ix,iy,iz) value
template <class T>
inline T const& TArray<T>::operator()(uint_4 ix, uint_4 iy, uint_4 iz) const
{
#ifdef SO_BOUNDCHECKING
  CheckBound(ix, iy, iz, 0, 0, 4);
#endif
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2]) );
}

//! Return element (ix,iy,iz) value
template <class T>
inline T & TArray<T>::operator()(uint_4 ix, uint_4 iy, uint_4 iz) 
{
#ifdef SO_BOUNDCHECKING
  CheckBound(ix, iy, iz, 0, 0, 4);
#endif
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2]) );
}

//! Operator () : return element (ix,iy,iz,it,iu) value
template <class T>
inline T const& TArray<T>::operator()(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu) const
{
#ifdef SO_BOUNDCHECKING
  CheckBound(ix, iy, iz, it, iu, 4);
#endif
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2] +
	      it*step_[3] + iu*step_[4]) );
}

//! Operator () : return element (ix,iy,iz,it,iu) value
template <class T>
inline T & TArray<T>::operator()(uint_4 ix, uint_4 iy, uint_4 iz, uint_4 it, uint_4 iu)
{
#ifdef SO_BOUNDCHECKING
  CheckBound(ix, iy, iz, it, iu, 4);
#endif
  return ( *( mNDBlock.Begin()+ offset_+ 
	      ix*step_[0] + iy*step_[1] + iz*step_[2] +
	      it*step_[3] + iu*step_[4]) );
}


//! Operator [] : return element at positon ip
template <class T>
inline T const& TArray<T>::operator[](uint_8 ip) const
{
#ifdef SO_BOUNDCHECKING
  if (ip >= totsize_)  throw( ParmError("TArray<T>::operator[] Out-of-bound Error") );
#endif
return *(mNDBlock.Begin()+Offset(ip));
}

//! Operator [] : return element at positon ip
template <class T>
inline T & TArray<T>::operator[](uint_8 ip) 
{
#ifdef SO_BOUNDCHECKING
  if (ip >= totsize_)  throw( ParmError("TArray<T>::operator[] Out-of-bound Error") );
#endif
return *(mNDBlock.Begin()+Offset(ip));
}


//! Return the value of first element
template <class T>
inline T TArray<T>::toScalar()
{ 
  if (Size() != 1) throw(SzMismatchError("TArray<T>::operator T() Size() != 1")) ;
  return ( (*this)[0] );
}

// Typedef pour simplifier
/*! \ingroup TArray
  \typedef Array
  \brief To simplified TArray<r_8> writing
*/
typedef TArray<r_8> Array;

} // Fin du namespace

#endif
